<?php
// public/setores/novo.php
require_once __DIR__ . '/../../app/core/auth.php';
require_login();

require_once __DIR__ . '/../../app/core/rbac.php';
require_role('admin'); // 🔒 somente admin

require_once __DIR__ . '/../../app/config/db.php';
require_once __DIR__ . '/../../app/core/helpers.php';

$pageTitle = 'Novo Setor';

/* Empresas ativas para o select */
$empresas = $pdo->query("
  SELECT id, razao_social
    FROM empresas
   WHERE status='ativo'
   ORDER BY razao_social
")->fetchAll();

/* Estado do form */
$errors = [];
$old = [
  'empresa_id'  => $_POST['empresa_id']  ?? '',
  'gerencia_id' => $_POST['gerencia_id'] ?? '',
  'nome'        => $_POST['nome']        ?? '',
  'observacao'  => $_POST['observacao']  ?? '',
];

/* POST */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $empresa_id  = (int)($_POST['empresa_id'] ?? 0);
  $gerencia_id = (int)($_POST['gerencia_id'] ?? 0);
  $nome        = trim($_POST['nome'] ?? '');
  $observacao  = trim($_POST['observacao'] ?? '');

  // validações
  if ($empresa_id <= 0) {
    $errors[] = 'Selecione a empresa.';
  } else {
    $chkEmp = $pdo->prepare("SELECT COUNT(*) FROM empresas WHERE id=? AND status='ativo'");
    $chkEmp->execute([$empresa_id]);
    if (!(int)$chkEmp->fetchColumn()) {
      $errors[] = 'Empresa inválida ou inativa.';
    }
  }

  if ($gerencia_id <= 0) {
    $errors[] = 'Selecione a gerência.';
  } else {
    // gerência precisa existir, estar ativa e pertencer à MESMA empresa
    $chkGer = $pdo->prepare("SELECT COUNT(*) FROM gerencias WHERE id=? AND empresa_id=? AND status='ativo'");
    $chkGer->execute([$gerencia_id, $empresa_id]);
    if (!(int)$chkGer->fetchColumn()) {
      $errors[] = 'Gerência inválida para a empresa selecionada.';
    }
  }

  if ($nome === '') {
    $errors[] = 'Nome do Setor é obrigatório.';
  }

  if (!$errors) {
    // se sua tabela tiver 'criado_em', você pode adicionar NOW() no INSERT
    $ins = $pdo->prepare("
      INSERT INTO setores (empresa_id, gerencia_id, nome, observacao, status)
      VALUES (?, ?, ?, ?, 'ativo')
    ");
    $ok = $ins->execute([$empresa_id, $gerencia_id, $nome, $observacao !== '' ? $observacao : null]);

    if ($ok) {
      redirect('setores/index.php');
    } else {
      $errors[] = 'Não foi possível salvar o setor.';
    }
  }
}

include __DIR__ . '/../../app/partials/head.php';
include __DIR__ . '/../../app/partials/topbar.php';
?>

<div class="d-flex justify-content-between align-items-center mb-3">
  <h1 class="h4 m-0">Novo setor</h1>
  <a href="<?= e(url('setores/index.php')) ?>" class="btn btn-outline-secondary">Voltar</a>
</div>

<?php if ($errors): ?>
  <div class="alert alert-danger">
    <strong>Não foi possível salvar:</strong><br>
    <?= implode('<br>', array_map('e', $errors)) ?>
  </div>
<?php endif; ?>

<div class="card">
  <div class="card-body">
    <form method="post" id="formSetor" class="row g-3">
      <div class="col-12 col-md-6">
        <label class="form-label">Empresa</label>
        <select name="empresa_id" id="empresa" class="form-select" required>
          <option value="">Selecione</option>
          <?php foreach($empresas as $e): ?>
            <option value="<?= (int)$e['id'] ?>" <?= ((string)$old['empresa_id'] === (string)$e['id']) ? 'selected' : '' ?>>
              <?= e($e['razao_social']) ?>
            </option>
          <?php endforeach; ?>
        </select>
      </div>

      <div class="col-12 col-md-6">
        <label class="form-label">Gerência</label>
        <select name="gerencia_id" id="gerencia" class="form-select" required>
          <option value=""><?= $old['empresa_id'] ? 'Carregando...' : 'Selecione a empresa primeiro' ?></option>
        </select>
      </div>

      <div class="col-12">
        <label class="form-label">Nome do Setor</label>
        <input name="nome" class="form-control" required value="<?= e($old['nome']) ?>">
      </div>

      <div class="col-12">
        <label class="form-label">Ger. Administrativo <span class="text-muted">(opcional)</span></label>
        <input name="observacao" class="form-control" value="<?= e($old['observacao']) ?>">
      </div>

      <div class="col-12">
        <button class="btn btn-primary">Salvar</button>
        <a href="<?= e(url('setores/index.php')) ?>" class="btn btn-outline-secondary">Cancelar</a>
      </div>
    </form>
  </div>
</div>

<script>
async function carregarGerenciasPorEmpresa(empresaId, selectedId=0){
  const gSel = document.getElementById('gerencia');
  gSel.innerHTML = '<option value="">' + (empresaId ? 'Carregando...' : 'Selecione a empresa primeiro') + '</option>';
  if(!empresaId){ return; }
  try {
    const res = await fetch('<?= e(url('api/gerencias_by_empresa.php')) ?>?empresa_id='+encodeURIComponent(empresaId));
    const js  = await res.json();
    gSel.innerHTML = '<option value="">Selecione</option>';
    js.forEach(g=>{
      const opt=document.createElement('option');
      opt.value=g.id; opt.textContent=g.nome;
      if (Number(selectedId)===Number(g.id)) opt.selected = true;
      gSel.appendChild(opt);
    });
  } catch(e){
    gSel.innerHTML = '<option value="">Erro ao carregar</option>';
  }
}

document.getElementById('empresa').addEventListener('change', e=>carregarGerenciasPorEmpresa(e.target.value));

// Se já veio empresa pré-selecionada (pós-erro), carrega as gerências mantendo a seleção anterior
window.addEventListener('DOMContentLoaded', ()=>{
  const emp = '<?= (int)$old['empresa_id'] ?>';
  const ger = '<?= (int)$old['gerencia_id'] ?>';
  if (emp && Number(emp) > 0) carregarGerenciasPorEmpresa(emp, ger);
});
</script>

<?php include __DIR__ . '/../../app/partials/foot.php'; ?>
