<?php
// public/setores/index.php
require_once __DIR__ . '/../../app/core/auth.php';
require_login();

require_once __DIR__ . '/../../app/core/rbac.php';
require_role('admin'); // 🔒 somente admin

require_once __DIR__ . '/../../app/config/db.php';
require_once __DIR__ . '/../../app/core/helpers.php';

$pageTitle = 'Setores';

/* Filtros */
$q          = trim($_GET['q'] ?? '');
$empresaId  = (int)($_GET['empresa_id'] ?? 0);
$gerenciaId = (int)($_GET['gerencia_id'] ?? 0);
$status     = $_GET['status'] ?? '';
$page       = max(1, (int)($_GET['p'] ?? 1));
$pp         = 10;
$off        = ($page - 1) * $pp;

/* Ação: Inativar */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['inativar_id'])) {
  $sid = (int)$_POST['inativar_id'];
  $st  = $pdo->prepare("UPDATE setores SET status='inativo' WHERE id=?");
  $st->execute([$sid]);
  header("Location: " . url('setores/index.php'));
  exit;
}

/* Empresas ativas para o filtro */
$empresas = $pdo->query("
  SELECT id, razao_social
    FROM empresas
   WHERE status='ativo'
   ORDER BY razao_social
")->fetchAll();

/* Montagem dinâmica de WHERE */
$params = [];
$where  = "WHERE 1=1";
if ($q !== '') { $where .= " AND s.nome LIKE :q"; $params[':q'] = "%{$q}%"; }
if ($empresaId > 0) { $where .= " AND s.empresa_id=:e"; $params[':e'] = $empresaId; }
if ($gerenciaId > 0){ $where .= " AND s.gerencia_id=:g"; $params[':g'] = $gerenciaId; }
if ($status === 'ativo' || $status === 'inativo'){ $where .= " AND s.status=:s"; $params[':s'] = $status; }

/* Contagem e listagem */
$sqlCount = $pdo->prepare("SELECT COUNT(*) c FROM setores s $where");
$sqlCount->execute($params);
$rows = (int)$sqlCount->fetch()['c'];

$sql = $pdo->prepare("
  SELECT s.*, e.razao_social, g.nome AS gerencia_nome
    FROM setores s
    JOIN empresas e  ON e.id = s.empresa_id
    JOIN gerencias g ON g.id = s.gerencia_id
    $where
   ORDER BY s.criado_em DESC, s.id DESC
   LIMIT $pp OFFSET $off
");
$sql->execute($params);
$data = $sql->fetchAll();

/* View */
include __DIR__ . '/../../app/partials/head.php';
include __DIR__ . '/../../app/partials/topbar.php';
?>
<div class="d-flex justify-content-between align-items-center mb-3">
  <h1 class="h4 m-0">Setores</h1>
  <a href="<?= e(url('setores/novo.php')) ?>" class="btn btn-primary">Novo setor</a>
</div>

<form class="row g-2 mb-3" id="filtros">
  <div class="col-12 col-md-3">
    <input type="text" name="q" class="form-control" placeholder="Buscar por nome" value="<?= e($q) ?>">
  </div>
  <div class="col-12 col-md-3">
    <select name="empresa_id" id="f_empresa" class="form-select">
      <option value="0">Todas as empresas</option>
      <?php foreach ($empresas as $e): ?>
        <option value="<?= (int)$e['id'] ?>" <?= $empresaId == $e['id'] ? 'selected' : '' ?>>
          <?= e($e['razao_social']) ?>
        </option>
      <?php endforeach; ?>
    </select>
  </div>
  <div class="col-12 col-md-3">
    <select name="gerencia_id" id="f_gerencia" class="form-select">
      <option value="0">Todas as gerências</option>
      <!-- opções serão carregadas via JS conforme a empresa -->
    </select>
  </div>
  <div class="col-6 col-md-2">
    <select name="status" class="form-select">
      <option value="">Status (todos)</option>
      <option value="ativo"   <?= $status==='ativo'?'selected':'' ?>>Ativo</option>
      <option value="inativo" <?= $status==='inativo'?'selected':'' ?>>Inativo</option>
    </select>
  </div>
  <div class="col-6 col-md-1">
    <button class="btn btn-outline-secondary w-100">Filtrar</button>
  </div>
</form>

<div class="table-responsive">
  <table class="table table-striped align-middle">
    <thead>
      <tr>
        <th>ID</th>
        <th>Empresa</th>
        <th>Gerência</th>
        <th>Nome</th>
        <th>Status</th>
        <th></th>
      </tr>
    </thead>
    <tbody>
      <?php if (!$data): ?>
        <tr>
          <td colspan="6" class="text-muted">Nenhum setor encontrado.</td>
        </tr>
      <?php else: ?>
        <?php foreach ($data as $r): ?>
          <tr>
            <td><?= (int)$r['id'] ?></td>
            <td><?= e($r['razao_social']) ?></td>
            <td><?= e($r['gerencia_nome']) ?></td>
            <td><?= e($r['nome']) ?></td>
            <td>
              <span class="badge bg-<?= $r['status']==='ativo' ? 'success' : 'secondary' ?>">
                <?= e($r['status']) ?>
              </span>
            </td>
            <td class="text-end">
              <a class="btn btn-sm btn-outline-primary" href="<?= e(url('setores/editar.php?id='.$r['id'])) ?>">Editar</a>
              <?php if ($r['status'] === 'ativo'): ?>
                <form method="post" class="d-inline" onsubmit="return confirm('Inativar este setor?');">
                  <input type="hidden" name="inativar_id" value="<?= (int)$r['id'] ?>">
                  <button class="btn btn-sm btn-outline-danger">Inativar</button>
                </form>
              <?php endif; ?>
            </td>
          </tr>
        <?php endforeach; ?>
      <?php endif; ?>
    </tbody>
  </table>
</div>

<?php
$pages = (int)ceil($rows / $pp);
if ($pages > 1): ?>
  <nav>
    <ul class="pagination">
      <?php for ($i = 1; $i <= $pages; $i++): ?>
        <li class="page-item <?= $i == $page ? 'active' : '' ?>">
          <a class="page-link"
             href="?p=<?= $i ?>&q=<?= urlencode($q) ?>&empresa_id=<?= $empresaId ?>&gerencia_id=<?= $gerenciaId ?>&status=<?= urlencode($status) ?>">
            <?= $i ?>
          </a>
        </li>
      <?php endfor; ?>
    </ul>
  </nav>
<?php endif; ?>

<script>
async function carregarGerencias(empresaId, selectedId=0) {
  const gSel = document.getElementById('f_gerencia');
  gSel.innerHTML = '<option value="0">Todas as gerências</option>';
  if (!empresaId || String(empresaId) === '0') return;
  try {
    const res = await fetch('<?= e(url('api/gerencias_by_empresa.php')) ?>?empresa_id=' + encodeURIComponent(empresaId));
    const js  = await res.json();
    js.forEach(g => {
      const opt = document.createElement('option');
      opt.value = g.id;
      opt.textContent = g.nome;
      if (Number(selectedId) === Number(g.id)) opt.selected = true;
      gSel.appendChild(opt);
    });
  } catch (e) {
    // opcional: console.error(e);
  }
}

// on change empresa -> recarrega gerências
const fEmpresa = document.getElementById('f_empresa');
fEmpresa.addEventListener('change', () => carregarGerencias(fEmpresa.value));

// carga inicial respeitando filtros atuais
window.addEventListener('DOMContentLoaded', () => {
  carregarGerencias('<?= (int)$empresaId ?>','<?= (int)$gerenciaId ?>');
});
</script>

<?php include __DIR__ . '/../../app/partials/foot.php'; ?>
