<?php
// public/relatorios/index.php
require_once __DIR__ . '/../../app/core/auth.php';
require_login();
require_once __DIR__ . '/../../app/config/db.php';
require_once __DIR__ . '/../../app/core/helpers.php';

// (Opcional) RBAC defensivo: bloqueia empresa_user
$rbac_loaded = @include_once __DIR__ . '/../../app/core/rbac.php';
if ($rbac_loaded && function_exists('current_role')) {
  $role = current_role();
  if (!in_array($role, ['admin','gerente','diretor'], true)) {
    http_response_code(403);
    include __DIR__ . '/../../app/partials/head.php';
    include __DIR__ . '/../../app/partials/topbar.php';
    echo "<div class='container mt-4'><div class='alert alert-danger'>Acesso negado aos relatórios consolidados.</div></div>";
    include __DIR__ . '/../../app/partials/foot.php';
    exit;
  }
}

$pageTitle = 'Relatórios Consolidados';
include __DIR__ . '/../../app/partials/head.php';
include __DIR__ . '/../../app/partials/topbar.php';

// ---------------- Parâmetros ----------------
$empresaId   = (int)($_GET['empresa_id'] ?? 0);
$checklistId = (int)($_GET['checklist_id'] ?? 0);

// Paginação
$pp  = max(5, min(50, (int)($_GET['pp'] ?? 10)));
$p   = max(1, (int)($_GET['p'] ?? 1));
$off = ($p - 1) * $pp;

// ---------------- Helpers ----------------
function qs(array $extra = []): string {
  $cur = $_GET;
  foreach ($extra as $k=>$v) {
    if ($v === null) unset($cur[$k]); else $cur[$k] = $v;
  }
  return http_build_query($cur);
}
function render_pagination_simple(int $current, int $totalPages) {
  if ($totalPages <= 1) return;
  $base = strtok($_SERVER['REQUEST_URI'], '?');

  $ini = max(1, $current - 1);
  $fim = min($totalPages, $current + 1);

  echo '<nav aria-label="Paginação"><ul class="pagination justify-content-center">';
  $prev = max(1, $current - 1);
  echo '<li class="page-item'.($current<=1?' disabled':'').'"><a class="page-link" href="'.e($base.'?'.qs(['p'=>$prev])).'">Anterior</a></li>';
  for ($i=$ini; $i<=$fim; $i++) {
    echo '<li class="page-item'.($i===$current?' active':'').'"><a class="page-link" href="'.e($base.'?'.qs(['p'=>$i])).'">'.(int)$i.'</a></li>';
  }
  $next = min($totalPages, $current + 1);
  echo '<li class="page-item'.($current>=$totalPages?' disabled':'').'"><a class="page-link" href="'.e($base.'?'.qs(['p'=>$next])).'">Próxima</a></li>';
  echo '</ul></nav>';
}

// ---------------- Empresas (select) ----------------
// SIMPLIFICADO: só razao_social para evitar "Unknown column"
$empresas = [];
try {
  $empresas = $pdo->query("
    SELECT id, razao_social AS nome
    FROM empresas
    ORDER BY razao_social
  ")->fetchAll(PDO::FETCH_ASSOC);
} catch (Throwable $ex) {
  echo "<div class='alert alert-danger'>Erro ao carregar empresas: ".e($ex->getMessage())."</div>";
}

// ---------------- Checklists (após escolher empresa) ----------------
$checklistsOptions = [];
if ($empresaId > 0) {
  try {
    $st = $pdo->prepare("
      SELECT DISTINCT c.id, c.titulo
        FROM checklists c
        JOIN execucoes e
          ON e.checklist_id = c.id
         AND e.empresa_id   = c.empresa_id
       WHERE c.empresa_id = ?
         AND e.status = 'finalizado'
       ORDER BY c.titulo
    ");
    $st->execute([$empresaId]);
    $checklistsOptions = $st->fetchAll(PDO::FETCH_ASSOC);
  } catch (Throwable $ex) {
    echo "<div class='alert alert-danger'>Erro ao carregar checklists: ".e($ex->getMessage())."</div>";
  }
}

// ---------------- Execuções finalizadas + notas ----------------
$execs = [];
$total = 0;
$totalPages = 1;
$notas = [];

if ($empresaId > 0) {
  $where  = "WHERE e.status='finalizado' AND e.empresa_id = :emp";
  $params = [':emp' => $empresaId];
  if ($checklistId > 0) { $where .= " AND e.checklist_id = :chk"; $params[':chk'] = $checklistId; }

  try {
    // total
    $sqlCount = $pdo->prepare("
      SELECT COUNT(1)
        FROM execucoes e
        JOIN empresas  em ON em.id = e.empresa_id
        JOIN checklists c ON c.id  = e.checklist_id
      $where
    ");
    $sqlCount->execute($params);
    $total = (int)$sqlCount->fetchColumn();
    $totalPages = max(1, (int)ceil($total / $pp));

    // listagem
    $sql = $pdo->prepare("
      SELECT
        e.id, e.checklist_id, e.finalizado_em, e.iniciado_em,
        em.razao_social,
        c.titulo
      FROM execucoes e
      JOIN empresas  em ON em.id = e.empresa_id
      JOIN checklists c  ON c.id = e.checklist_id
      $where
      ORDER BY e.finalizado_em DESC, e.id DESC
      LIMIT :lim OFFSET :off
    ");
    foreach ($params as $k=>$v) { $sql->bindValue($k, $v); }
    $sql->bindValue(':lim', $pp, PDO::PARAM_INT);
    $sql->bindValue(':off', $off, PDO::PARAM_INT);
    $sql->execute();
    $execs = $sql->fetchAll(PDO::FETCH_ASSOC);

    // notas (0..100)
    $calcStmt = $pdo->prepare("
      SELECT p.peso, r.resposta
        FROM checklist_perguntas p
        JOIN respostas r ON r.pergunta_id = p.id
       WHERE r.execucao_id = ?
    ");
    foreach ($execs as $row) {
      $calcStmt->execute([(int)$row['id']]);
      $rows = $calcStmt->fetchAll(PDO::FETCH_ASSOC);
      $sumP = 0; $sumV = 0;
      foreach ($rows as $r) {
        $peso = (float)$r['peso'];
        $val  = $r['resposta'] === 'conforme' ? 1 : ($r['resposta'] === 'nao_conforme' ? 0 : null);
        if ($val === null) continue;
        $sumP += $peso;
        $sumV += $peso * $val;
      }
      $notas[$row['id']] = $sumP > 0 ? round(($sumV / $sumP) * 100, 2) : null;
    }
  } catch (Throwable $ex) {
    echo "<div class='alert alert-danger'>Erro ao carregar execuções: ".e($ex->getMessage())."</div>";
  }
}
?>
<style>
  .filters .form-select, .filters .form-control { min-width: 220px; }
  .card-table { border-radius: 1rem; overflow: hidden; }
  .muted { color: #6c757d; }
</style>

<div class="container my-4">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <h1 class="h4 m-0">Relatórios Consolidados</h1>
    <a class="btn btn-outline-secondary" href="<?= e(strtok($_SERVER['REQUEST_URI'], '?')) ?>">Limpar</a>
  </div>

  <!-- Filtros -->
  <form class="row gy-2 gx-2 align-items-end filters mb-3" method="get">
    <div class="col-12 col-md-5">
      <label class="form-label mb-1">Empresa</label>
      <select class="form-select" name="empresa_id" onchange="this.form.submit()">
        <option value="0">Selecione uma empresa… (inicialmente vazio)</option>
        <?php foreach ($empresas as $e): ?>
          <option value="<?= (int)$e['id'] ?>" <?= $empresaId===$e['id']?'selected':'' ?>>
            <?= e($e['nome']) ?>
          </option>
        <?php endforeach; ?>
      </select>
    </div>

    <div class="col-12 col-md-5">
      <label class="form-label mb-1">Checklist (somente com execuções finalizadas)</label>
      <select class="form-select" name="checklist_id" <?= $empresaId>0?'':'disabled' ?> >
        <option value="0">Todos os checklists finalizados dessa empresa</option>
        <?php foreach ($checklistsOptions as $c): ?>
          <option value="<?= (int)$c['id'] ?>" <?= $checklistId===$c['id']?'selected':'' ?>>
            <?= e($c['titulo']) ?>
          </option>
        <?php endforeach; ?>
      </select>
    </div>

    <div class="col-6 col-md-2">
      <label class="form-label mb-1">Itens/página</label>
      <select class="form-select" name="pp">
        <?php foreach ([10,20,30,50] as $opt): ?>
          <option value="<?= $opt ?>" <?= $pp==$opt?'selected':'' ?>><?= $opt ?></option>
        <?php endforeach; ?>
      </select>
    </div>

    <div class="col-12">
      <button class="btn btn-primary">Aplicar</button>
    </div>
  </form>

  <?php if ($empresaId === 0): ?>
    <div class="alert alert-info">
      Selecione uma empresa para listar as execuções finalizadas. A página inicia vazia por padrão.
    </div>
  <?php else: ?>
    <div class="card card-table">
      <div class="card-body">
        <div class="d-flex justify-content-between align-items-center mb-2">
          <h5 class="card-title m-0">Execuções finalizadas</h5>
          <div class="muted small">
            Página <?= (int)$p ?> de <?= (int)$totalPages ?> &middot; Total: <?= (int)$total ?>
          </div>
        </div>

        <div class="table-responsive">
          <table class="table table-striped table-sm align-middle">
            <thead>
              <tr>
                <th style="width:80px">ID</th>
                <th>Empresa</th>
                <th>Checklist</th>
                <th style="width:180px">Finalizado em</th>
                <th style="width:90px">Nota</th>
                <th style="width:160px">Ações</th>
              </tr>
            </thead>
            <tbody>
              <?php if (empty($execs)): ?>
                <tr><td colspan="6">Nenhuma execução encontrada para os filtros selecionados.</td></tr>
              <?php else: ?>
                <?php foreach ($execs as $ex): ?>
                  <?php
                    // usa o helper que você colocou em app/core/helpers.php
                    $okOri = orientacoes_concluidas($pdo, (int)$ex['id']);
                  ?>
                  <tr>
                    <td>#<?= (int)$ex['id'] ?></td>
                    <td><?= e($ex['razao_social']) ?></td>
                    <td><?= e($ex['titulo']) ?></td>
                    <td><?= e(date('d/m/Y H:i', strtotime($ex['finalizado_em']))) ?></td>
                    <td>
                      <?= isset($notas[$ex['id']]) && $notas[$ex['id']]!==null
                          ? e(number_format($notas[$ex['id']], 2, ',', '')).'%' : '—' ?>
                    </td>
                    <td class="text-end">
                      <?php if ($okOri): ?>
                        <a class="btn btn-sm btn-primary"
                           target="_blank"
                           href="<?= e(url('relatorios/pdf.php?execucao_id='.(int)$ex['id'])) ?>">
                          Gerar PDF
                        </a>
                      <?php else: ?>
                        <button class="btn btn-sm btn-outline-secondary" disabled
                                title="Finalize as Orientações técnicas desta execução para liberar o PDF.">
                          Gerar PDF
                        </button>
                        <a class="btn btn-sm btn-outline-dark"
                           href="<?= e(url('execucoes/orientacoes.php?execucao_id='.(int)$ex['id'])) ?>">
                          Orientação técnica
                        </a>
                      <?php endif; ?>
                    </td>
                  </tr>
                <?php endforeach; ?>
              <?php endif; ?>
            </tbody>
          </table>
        </div>

        <div class="mt-3">
          <?php render_pagination_simple($p, $totalPages); ?>
        </div>
      </div>
    </div>
  <?php endif; ?>
</div>

<?php include __DIR__ . '/../../app/partials/foot.php'; ?>
