<?php
// public/checklists/editar.php
require_once __DIR__ . '/../../app/core/auth.php';
require_login();

require_once __DIR__ . '/../../app/core/rbac.php';
require_role('admin'); // 🔒 admin

require_once __DIR__ . '/../../app/config/db.php';
require_once __DIR__ . '/../../app/core/helpers.php';

/* ---------- FLASH HELPERS ---------- */
if (session_status() === PHP_SESSION_NONE) session_start();
if (!function_exists('flash_add')) {
  function flash_add(string $type, string $msg): void {
    $_SESSION['__flash'][] = ['t'=>$type, 'm'=>$msg];
  }
}
if (!function_exists('flash_show')) {
  function flash_show(): void {
    if (!empty($_SESSION['__flash'])) {
      foreach ($_SESSION['__flash'] as $f) {
        $cls = [
          'success'=>'alert-success',
          'warning'=>'alert-warning',
          'danger'=>'alert-danger',
          'info'=>'alert-info',
        ][$f['t']] ?? 'alert-info';
        echo "<div class='alert {$cls}'>{$f['m']}</div>";
      }
      unset($_SESSION['__flash']);
    }
  }
}

/* --------- Carga do checklist --------- */
$id = (int)($_GET['id'] ?? 0);
$st = $pdo->prepare("
  SELECT c.*, e.razao_social
    FROM checklists c
    JOIN empresas e ON e.id = c.empresa_id
   WHERE c.id = ?
");
$st->execute([$id]);
$check = $st->fetch();
if (!$check) {
  $pageTitle = 'Checklist';
  include __DIR__ . '/../../app/partials/head.php';
  include __DIR__ . '/../../app/partials/topbar.php';
  echo "<div class='container mt-4'><div class='alert alert-danger'>Checklist não encontrado.</div></div>";
  include __DIR__ . '/../../app/partials/foot.php';
  exit;
}
$empresaId = (int)$check['empresa_id'];

$pageTitle = 'Montagem de Checklist';

/* ===== helpers ===== */
function sum_pesos_total(PDO $pdo, int $cid): int {
  $v = (float)$pdo->query("SELECT COALESCE(SUM(peso),0) s FROM checklist_perguntas WHERE checklist_id=".$cid)->fetch()['s'];
  return (int)round($v);
}
function sums_by_gerencia(PDO $pdo, int $cid): array {
  $q = $pdo->prepare("
    SELECT g.id AS gerencia_id, g.nome AS gerencia_nome, COALESCE(SUM(p.peso),0) AS soma
      FROM gerencias g
 LEFT JOIN checklist_perguntas p
        ON p.gerencia_id = g.id AND p.checklist_id = ?
     WHERE g.empresa_id = (SELECT empresa_id FROM checklists WHERE id=?)
  GROUP BY g.id, g.nome
  ORDER BY g.nome
  ");
  $q->execute([$cid, $cid]);
  $out = [];
  foreach ($q->fetchAll() as $r) {
    $out[(int)$r['gerencia_id']] = [
      'nome' => $r['gerencia_nome'],
      'soma' => (int)round($r['s'])
    ];
  }
  return $out;
}
function get_resp(PDO $pdo, int $pid) {
  $q=$pdo->prepare("
    SELECT c.id, c.nome, c.cargo
      FROM checklist_pergunta_responsaveis r
      JOIN colaboradores c ON c.id = r.colaborador_id
     WHERE r.pergunta_id = ?
  ORDER BY r.id
     LIMIT 1
  ");
  $q->execute([$pid]); return $q->fetch();
}
function go_tab_perguntas(int $id){
  redirect('checklists/editar.php?id='.$id.'&tab=perguntas');
}
function go_tab_execucoes(int $id){
  redirect('checklists/editar.php?id='.$id.'&tab=execucoes');
}
function rrmdir_safe(string $dir): void {
  if (!is_dir($dir)) return;
  $it = new RecursiveDirectoryIterator($dir, FilesystemIterator::SKIP_DOTS);
  $files = new RecursiveIteratorIterator($it, RecursiveIteratorIterator::CHILD_FIRST);
  foreach ($files as $file) {
    if ($file->isDir()) @rmdir($file->getRealPath());
    else @unlink($file->getRealPath());
  }
  @rmdir($dir);
}

/* ===== POST actions ===== */

/* Salvar dados (aba 1): alerta gerências ≠ 100; bloqueia ativação */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['salvar_basico'])) {
  $statusReq = $_POST['status'] ?? $check['status'];

  // somas por gerência
  $byG = sums_by_gerencia($pdo, $id);
  $pendencias = [];
  foreach ($byG as $info) {
    if ((int)$info['soma'] !== 100) {
      $pendencias[] = e($info['nome']).' = <strong>'.(int)$info['soma'].'</strong>';
    }
  }

  if ($statusReq === 'ativo' && !empty($pendencias)) {
    flash_add('danger', 'Não é possível ativar. Cada Gerência precisa somar <strong>100</strong>.<br>Pendências: '.implode(' | ', $pendencias));
    $statusReq = $check['status']; // mantém status anterior
  }

  $up = $pdo->prepare("UPDATE checklists SET titulo=?, descricao=?, status=? WHERE id=?");
  $up->execute([trim($_POST['titulo']), trim($_POST['descricao']), $statusReq, $id]);
  $check['titulo']=$_POST['titulo']; $check['descricao']=$_POST['descricao']; $check['status']=$statusReq;

  if (!empty($pendencias)) {
    flash_add('warning', 'Verifique as gerências com soma ≠ 100: '.implode(' | ', $pendencias));
  } else {
    flash_add('success', 'Todas as gerências estão com soma <strong>100</strong>.');
  }
}

/* Remover pergunta */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['del_pergunta_id'])) {
  $dp = $pdo->prepare("DELETE FROM checklist_perguntas WHERE id=? AND checklist_id=?");
  $dp->execute([(int)$_POST['del_pergunta_id'], $id]);
  flash_add('warning', 'Pergunta removida.');
  go_tab_perguntas($id);
}

/* Adicionar pergunta (peso inteiro, responsável obrigatório) — gera pergunta_uid */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['add_pergunta'])) {
  $ger  = (int)$_POST['gerencia_id'];
  $set  = (int)$_POST['setor_id'];
  $txt  = trim($_POST['texto'] ?? '');
  $peso = (int)($_POST['peso'] ?? 0);
  $resp = (int)$_POST['responsavel_id'];

  if (!$ger || !$set || $txt==='' || $peso<=0 || $resp<=0) {
    flash_add('danger', 'Preencha Gerência, Setor, Texto, <strong>Peso (inteiro &gt; 0)</strong> e <strong>Responsável</strong>.');
    go_tab_perguntas($id);
  }

  // soma atual da gerência
  $stS = $pdo->prepare("SELECT COALESCE(SUM(peso),0) s FROM checklist_perguntas WHERE checklist_id=? AND gerencia_id=?");
  $stS->execute([$id, $ger]);
  $somaAtual = (int)round((float)$stS->fetch()['s']);

  if ($somaAtual + $peso > 100) {
    flash_add('danger', "Não é possível adicionar: a soma na Gerência ficaria <strong>".($somaAtual + $peso)."</strong> (&gt; 100). Atual: <strong>{$somaAtual}</strong>.");
    go_tab_perguntas($id);
  }

  // INSERT com pergunta_uid
  $ins = $pdo->prepare("
    INSERT INTO checklist_perguntas (pergunta_uid, checklist_id, gerencia_id, setor_id, texto, peso)
    VALUES (UUID(), ?, ?, ?, ?, ?)
  ");
  $ins->execute([$id, $ger, $set, $txt, $peso]);
  $pid = (int)$pdo->lastInsertId();

  $pdo->prepare("INSERT INTO checklist_pergunta_responsaveis (pergunta_id, colaborador_id) VALUES (?, ?)")->execute([$pid, $resp]);

  $novaSoma = $somaAtual + $peso;
  if ($novaSoma < 100) {
    flash_add('warning', "Pergunta adicionada. <strong>Soma da Gerência</strong>: <strong>{$novaSoma}</strong> (abaixo de 100).");
  } else {
    flash_add('success', "Pergunta adicionada. Gerência fechou em <strong>100</strong>. ✅");
  }
  go_tab_perguntas($id);
}

/* Editar pergunta (modal) — preserva pergunta_uid */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['edit_pergunta'])) {
  $pid   = (int)$_POST['pergunta_id'];
  $txt   = trim($_POST['texto_edit'] ?? '');
  $pesoN = (int)($_POST['peso_edit'] ?? 0);
  $resp  = (int)$_POST['responsavel_id_edit'] ?? 0;
  $gerN  = (int)$_POST['gerencia_id_edit'];
  $setN  = (int)$_POST['setor_id_edit'];

  if ($pid<=0 || $txt==='' || $pesoN<=0 || $resp<=0 || $gerN<=0 || $setN<=0) {
    flash_add('danger', 'Para editar: Texto, <strong>Peso &gt; 0</strong>, <strong>Responsável</strong>, <strong>Gerência</strong> e <strong>Setor</strong> são obrigatórios.');
    go_tab_perguntas($id);
  }

  // dados atuais
  $qOld = $pdo->prepare("SELECT gerencia_id AS gerO, setor_id AS setO, peso AS pesoO FROM checklist_perguntas WHERE id=? AND checklist_id=?");
  $qOld->execute([$pid, $id]);
  $old = $qOld->fetch();
  if (!$old) { flash_add('danger','Pergunta não encontrada.'); go_tab_perguntas($id); }
  $gerO=(int)$old['gerO']; $setO=(int)$old['setO']; $pesoO=(int)$old['pesoO'];

  // somas por gerência sem a pergunta atual
  $stG = $pdo->prepare("SELECT gerencia_id, COALESCE(SUM(peso),0) s FROM checklist_perguntas WHERE checklist_id=? GROUP BY gerencia_id");
  $stG->execute([$id]);
  $somas = []; foreach($stG->fetchAll() as $r){ $somas[(int)$r['gerencia_id']] = (int)round((float)$r['s']); }
  $somas[$gerO] = ($somas[$gerO] ?? 0) - $pesoO; if ($somas[$gerO] < 0) $somas[$gerO]=0;
  $somas[$gerN] = ($somas[$gerN] ?? 0) + $pesoN;

  $estouro = [];
  foreach ($somas as $gId=>$s) { if ($s > 100) $estouro[] = $gId.'='.$s; }
  if ($estouro) {
    $nomes = $pdo->query("SELECT id,nome FROM gerencias WHERE empresa_id=". (int)$empresaId)->fetchAll(PDO::FETCH_KEY_PAIR);
    $msgs = [];
    foreach ($somas as $gId=>$s) if ($s>100) $msgs[] = ($nomes[$gId] ?? "Gerência {$gId}") . ' = ' . $s;
    flash_add('danger', 'Não é possível salvar: alguma Gerência ficará &gt; 100. <br><strong>'.implode(' | ', $msgs).'</strong>');
    go_tab_perguntas($id);
  }

  // salvar
  $up = $pdo->prepare("UPDATE checklist_perguntas SET texto=?, peso=?, gerencia_id=?, setor_id=? WHERE id=? AND checklist_id=?");
  $up->execute([$txt, $pesoN, $gerN, $setN, $pid, $id]);

  $pdo->prepare("DELETE FROM checklist_pergunta_responsaveis WHERE pergunta_id=?")->execute([$pid]);
  $pdo->prepare("INSERT INTO checklist_pergunta_responsaveis (pergunta_id, colaborador_id) VALUES (?, ?)")->execute([$pid, $resp]);

  // feedback das gerências envolvidas
  $msgs = [];
  $chkG = array_unique([$gerO, $gerN]);
  $stOne = $pdo->prepare("SELECT COALESCE(SUM(peso),0) s FROM checklist_perguntas WHERE checklist_id=? AND gerencia_id=?");
  foreach ($chkG as $gTest) {
    if (!$gTest) continue;
    $stOne->execute([$id, $gTest]);
    $s = (int)round((float)$stOne->fetch()['s']);
    $gNome = $pdo->query("SELECT nome FROM gerencias WHERE id=".(int)$gTest)->fetchColumn();
    $msgs[] = e($gNome).": <strong>{$s}</strong>";
  }
  flash_add('success', 'Pergunta atualizada. Somas: '.implode(' | ', $msgs).'.');
  go_tab_perguntas($id);
}

/* ===== EXECUÇÕES: liberar & cancelar liberadas ===== */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['liberar_execucao'])) {
  // cria uma execução LIBERADA (sem dono e sem datas)
  $ins = $pdo->prepare("
    INSERT INTO execucoes (checklist_id, empresa_id, executado_por, status, iniciado_em, finalizado_em)
    VALUES (?, ?, NULL, 'liberada', NULL, NULL)
  ");
  if ($ins->execute([$id, $empresaId])) {
    $novoId = (int)$pdo->lastInsertId();
    flash_add('success', 'Execução #'.$novoId.' liberada com sucesso.');
  } else {
    flash_add('danger', 'Não foi possível liberar a execução.');
  }
  go_tab_execucoes($id);
}

if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['cancelar_execucao'])) {
  $execId = (int)($_POST['cancelar_execucao'] ?? 0);

  // Só pode cancelar se ainda estiver liberada e sem datas
  $st = $pdo->prepare("
    SELECT id FROM execucoes
     WHERE id=? AND checklist_id=? AND empresa_id=? AND status='liberada' AND iniciado_em IS NULL AND finalizado_em IS NULL
     LIMIT 1
  ");
  $st->execute([$execId, $id, $empresaId]);
  $can = $st->fetchColumn();

  if (!$can) {
    flash_add('warning', 'Não foi possível cancelar: execução não está mais em estado "liberada" ou não pertence a este checklist.');
    go_tab_execucoes($id);
  }

  // Apagar dependências: resposta_fotos, respostas
  try {
    $pdo->beginTransaction();

    // apagar fotos (arquivos e registros)
    $q = $pdo->prepare("
      SELECT rf.id, rf.caminho
        FROM resposta_fotos rf
        JOIN respostas r ON r.id = rf.resposta_id
       WHERE r.execucao_id = ?
    ");
    $q->execute([$execId]);
    $fotos = $q->fetchAll();

    // Apaga registros
    $pdo->prepare("
      DELETE rf FROM resposta_fotos rf
      JOIN respostas r ON r.id = rf.resposta_id
      WHERE r.execucao_id = ?
    ")->execute([$execId]);
    $pdo->prepare("DELETE FROM respostas WHERE execucao_id = ?")->execute([$execId]);

    // Apaga a execução
    $pdo->prepare("DELETE FROM execucoes WHERE id = ?")->execute([$execId]);

    $pdo->commit();

    // Remove pasta de uploads desta execução (se existir)
    $upDir = realpath(__DIR__ . '/../../'); // .../public
    $execDir = $upDir . '/uploads/execucoes/' . $execId;
    rrmdir_safe($execDir);

    flash_add('success', 'Execução #'.$execId.' cancelada e todas as dependências foram removidas.');
  } catch (Throwable $e) {
    $pdo->rollBack();
    flash_add('danger', 'Falha ao cancelar a execução: '.$e->getMessage());
  }

  go_tab_execucoes($id);
}

/* ===== cargas para render ===== */
$gerencias = $pdo->prepare("SELECT id, nome FROM gerencias WHERE empresa_id=? AND status='ativo' ORDER BY nome");
$gerencias->execute([$empresaId]);
$gerencias = $gerencias->fetchAll();

$colabs = $pdo->prepare("SELECT id, nome, cargo FROM colaboradores WHERE empresa_id=? AND status='ativo' ORDER BY nome");
$colabs->execute([$empresaId]);
$colabs = $colabs->fetchAll();

$perg = $pdo->prepare("
  SELECT p.*, g.nome AS gerencia_nome, s.nome AS setor_nome
    FROM checklist_perguntas p
    JOIN gerencias g ON g.id = p.gerencia_id
    JOIN setores   s ON s.id = p.setor_id
   WHERE p.checklist_id = ?
ORDER BY g.nome, s.nome, p.id
");
$perg->execute([$id]);
$perguntas = $perg->fetchAll();

/* somas */
$sumTotalInt = sum_pesos_total($pdo, $id);
$byGer = sums_by_gerencia($pdo, $id);

/* agrupar por gerencia/setor */
$tree = [];
foreach ($perguntas as $p) {
  $tree[$p['gerencia_id'].'|'.$p['gerencia_nome']] = $tree[$p['gerencia_id'].'|'.$p['gerencia_nome']] ?? [];
  $tree[$p['gerencia_id'].'|'.$p['gerencia_nome']][$p['setor_id'].'|'.$p['setor_nome']] = $tree[$p['gerencia_id'].'|'.$p['gerencia_nome']][$p['setor_id'].'|'.$p['setor_nome']] ?? [];
  $tree[$p['gerencia_id'].'|'.$p['gerencia_nome']][$p['setor_id'].'|'.$p['setor_nome']][] = $p;
}

/* execuções desse checklist */
$execs = $pdo->prepare("
  SELECT id, status, executado_por, iniciado_em, finalizado_em
    FROM execucoes
   WHERE checklist_id = ? AND empresa_id = ?
ORDER BY 
  CASE status
    WHEN 'liberada' THEN 0
    WHEN 'em_andamento' THEN 1
    WHEN 'finalizado' THEN 2
    ELSE 3
  END, id DESC
");
$execs->execute([$id, $empresaId]);
$execucoes = $execs->fetchAll();

$activeTab = in_array(($_GET['tab'] ?? ''), ['perguntas','execucoes']) ? $_GET['tab'] : 'dados';

include __DIR__ . '/../../app/partials/head.php';
include __DIR__ . '/../../app/partials/topbar.php';
?>
<style>
.table tr.row-gerencia > td { background-color:#0B3040!important; color:#fff!important; font-weight:700; }
.table tr.row-setor    > td { background-color:#F1A983!important; color:#000!important; font-weight:700; }
.badge-status { text-transform:uppercase }
</style>

<div class="d-flex justify-content-between align-items-center mb-3">
  <h1 class="h4 m-0">Checklist: <?= e($check['titulo']) ?></h1>
  <a class="btn btn-outline-secondary" href="<?= e(url('checklists/index.php')) ?>">Voltar</a>
</div>

<?php flash_show(); ?>

<ul class="nav nav-tabs" id="tabChecklist" role="tablist">
  <li class="nav-item" role="presentation">
    <button class="nav-link <?= $activeTab==='dados'?'active':'' ?>" id="tab-dados" data-bs-toggle="tab" data-bs-target="#pane-dados" type="button" role="tab">Dados</button>
  </li>
  <li class="nav-item" role="presentation">
    <button class="nav-link <?= $activeTab==='perguntas'?'active':'' ?>" id="tab-perguntas" data-bs-toggle="tab" data-bs-target="#pane-perguntas" type="button" role="tab">Perguntas</button>
  </li>
  <li class="nav-item" role="presentation">
    <button class="nav-link <?= $activeTab==='execucoes'?'active':'' ?>" id="tab-execs" data-bs-toggle="tab" data-bs-target="#pane-execs" type="button" role="tab">Execuções</button>
  </li>
</ul>

<div class="tab-content border border-top-0 p-3">
  <!-- TAB 1 -->
  <div class="tab-pane fade <?= $activeTab==='dados'?'show active':'' ?>" id="pane-dados" role="tabpanel">
    <div class="card mb-3">
      <div class="card-body">
        <form method="post" class="row g-3">
          <div class="col-12 col-md-6">
            <label class="form-label">Empresa</label>
            <input class="form-control" value="<?= e($check['razao_social']) ?>" disabled>
          </div>
          <div class="col-12 col-md-3">
            <label class="form-label">Status</label>
            <?php
              $todas100 = true;
              foreach ($byGer as $gInfo) { if ((int)$gInfo['soma'] !== 100) { $todas100=false; break; } }
            ?>
            <select name="status" class="form-select" <?= $todas100 ? '' : 'title="Para ativar, todas as gerências devem somar 100"' ?> >
              <option value="rascunho" <?= $check['status']==='rascunho'?'selected':'' ?>>Rascunho</option>
              <option value="ativo"    <?= $check['status']==='ativo'?'selected':'' ?> <?= $todas100?'':'disabled' ?>>Ativo</option>
              <option value="inativo"  <?= $check['status']==='inativo'?'selected':'' ?>>Inativo</option>
            </select>
            <div class="form-text">
              Somas por Gerência:
              <?php foreach ($byGer as $info): ?>
                <span class="badge <?= (int)$info['soma']===100?'bg-success':'bg-warning text-dark' ?> me-1">
                  <?= e($info['nome']) ?>: <?= (int)$info['soma'] ?>
                </span>
              <?php endforeach; ?>
            </div>
          </div>
          <div class="col-12 col-md-9">
            <label class="form-label">Título</label>
            <input name="titulo" class="form-control" value="<?= e($check['titulo']) ?>" required>
          </div>
          <div class="col-12">
            <label class="form-label">Objetivo</label>
            <textarea name="descricao" class="form-control" rows="3"><?= e($check['descricao']) ?></textarea>
          </div>
          <div class="col-12">
            <button class="btn btn-primary" name="salvar_basico" value="1">Salvar</button>
            <?php if (!$todas100): ?>
              <span class="badge bg-warning text-dark ms-2">Atenção: existem gerências ≠ 100</span>
            <?php else: ?>
              <span class="badge bg-success ms-2">Todas as gerências em 100</span>
            <?php endif; ?>
          </div>
        </form>
      </div>
    </div>
  </div>

  <!-- TAB 2 -->
  <div class="tab-pane fade <?= $activeTab==='perguntas'?'show active':'' ?>" id="pane-perguntas" role="tabpanel">
    <div class="alert alert-info">
      <strong>Regra:</strong> a soma dos Pesos deve ser <strong>100 por Gerência</strong>.
    </div>

    <!-- Adicionar -->
    <div class="card mb-4">
      <div class="card-body">
        <h5 class="card-title mb-3">Adicionar pergunta</h5>
        <form method="post" class="row g-3" id="formPergunta">
          <input type="hidden" name="add_pergunta" value="1">

          <div class="col-12 col-md-4">
            <label class="form-label">Gerência</label>
            <select name="gerencia_id" id="gerencia" class="form-select" required>
              <option value="">Selecione</option>
              <?php foreach($gerencias as $g): ?>
                <option value="<?= (int)$g['id'] ?>"><?= e($g['nome']) ?></option>
              <?php endforeach; ?>
            </select>
          </div>

          <div class="col-12 col-md-4">
            <label class="form-label">Setor</label>
            <select name="setor_id" id="setor" class="form-select" required>
              <option value="">Selecione a gerência</option>
            </select>
          </div>

          <div class="col-12">
            <label class="form-label">Texto da pergunta</label>
            <div class="input-group">
              <textarea name="texto" class="form-control" rows="4" required></textarea>
              <span class="input-group-text">Peso</span>
              <input type="number" step="1" min="1" name="peso" id="peso" class="form-control" value="1" required>
            </div>
          </div>

          <div class="col-12 col-md-6">
            <label class="form-label">Responsável no setor <span class="text-danger">*</span></label>
            <select name="responsavel_id" class="form-select" required>
              <option value="">Selecione</option>
              <?php foreach($colabs as $c): ?>
                <option value="<?= (int)$c['id'] ?>"><?= e(($c['cargo'] ? $c['cargo'].' - ' : '') . $c['nome']) ?></option>
              <?php endforeach; ?>
            </select>
          </div>

          <div class="col-12">
            <button class="btn btn-success">Adicionar pergunta</button>
          </div>
        </form>
      </div>
    </div>

    <!-- Estrutura -->
    <div class="card">
      <div class="card-body">
        <h5 class="card-title mb-3">Estrutura das perguntas</h5>
        <?php
        if (!$tree) {
          echo "<div class='text-muted'>Nenhuma pergunta adicionada.</div>";
        } else {
          $gIdx=0;
          echo "<div class='table-responsive'><table class='table table-sm align-middle'>";
          echo "<thead><tr>
                  <th style='width:90px'>#</th>
                  <th>Item</th>
                  <th>Responsável no setor</th>
                  <th style='width:100px'>Peso</th>
                  <th class='text-end' style='width:260px'>Ações</th>
                </tr></thead><tbody>";
          foreach ($tree as $gKey => $setores) {
            [$gId, $gNome] = explode('|', $gKey, 2);
            $gIdx++;
            $somaG = (int)($byGer[(int)$gId]['soma'] ?? 0);
            $badgeClass = $somaG===100 ? 'bg-success' : 'bg-warning text-dark';
            echo "<tr class='row-gerencia'>".
                   "<td><strong>{$gIdx}</strong></td>".
                   "<td><strong>Gerência:</strong> ".e($gNome)." ".
                   "<td></td><td></td><td></td>".
                 "</tr>";

            $sIdx=0;
            foreach ($setores as $sKey => $qs) {
              [$sId, $sNome] = explode('|', $sKey, 2);
              $sIdx++;
              $prefix = $gIdx.'.'.$sIdx;
              echo "<tr class='row-setor'>".
                     "<td><strong>{$prefix}</strong></td>".
                     "<td><strong>Setor:</strong> ".e($sNome)."</td>".
                     "<td></td><td></td><td></td>".
                   "</tr>";
              $qIdx=0;
              foreach ($qs as $p) {
                $qIdx++;
                $code = $prefix.'.'.$qIdx;
                $resp = get_resp($pdo, (int)$p['id']);
                $respTxt = $resp ? e(($resp['cargo']? $resp['cargo'].' - ' : '').$resp['nome']) : '<span class="text-muted">—</span>';
                $respId  = $resp['id'] ?? 0;

                echo "<tr>";
                echo   "<td>{$code}</td>";
                echo   "<td>".nl2br(e($p['texto']))."</td>";
                echo   "<td>{$respTxt}</td>";
                echo   "<td>".(int)$p['peso']."</td>";
                echo   "<td class='text-end'>
                          <button
                            class='btn btn-sm btn-outline-primary me-1 edit-btn'
                            data-bs-toggle='modal'
                            data-bs-target='#modalEdit'
                            data-pid='".(int)$p['id']."'
                            data-texto=\"".htmlspecialchars($p['texto'], ENT_QUOTES, 'UTF-8')."\"
                            data-peso='".(int)$p['peso']."'
                            data-resp='".(int)$respId."'
                            data-ger='".(int)$p['gerencia_id']."'
                            data-set='".(int)$p['setor_id']."'
                          >Editar</button>
                          <form method='post' onsubmit=\"return confirm('Remover esta pergunta?');\" class='d-inline'>
                            <input type='hidden' name='del_pergunta_id' value='".(int)$p['id']."'>
                            <button class='btn btn-sm btn-outline-danger'>Remover</button>
                          </form>
                        </td>";
                echo "</tr>";
              }
            }
          }
          echo "</tbody></table></div>";
        }
        ?>
      </div>
    </div>
  </div>

  <!-- TAB 3: Execuções -->
  <div class="tab-pane fade <?= $activeTab==='execucoes'?'show active':'' ?>" id="pane-execs" role="tabpanel">
    <div class="d-flex justify-content-between align-items-center mb-3">
      <h5 class="m-0">Execuções deste checklist</h5>
      <form method="post" class="m-0">
        <input type="hidden" name="liberar_execucao" value="1">
        <button class="btn btn-success">Liberar nova execução</button>
      </form>
    </div>
    <?php if (!$execucoes): ?>
      <div class="alert alert-info">Nenhuma execução criada ainda.</div>
    <?php else: ?>
      <div class="table-responsive">
        <table class="table table-sm align-middle">
          <thead><tr>
            <th>#</th>
            <th>Status</th>
            <th>Executado por</th>
            <th>Iniciado</th>
            <th>Finalizado</th>
            <th class="text-end">Ações</th>
          </tr></thead>
          <tbody>
          <?php foreach ($execucoes as $x): ?>
            <tr>
              <td><?= (int)$x['id'] ?></td>
              <td>
                <span class="badge badge-status bg-<?=
                  $x['status']==='liberada'?'warning text-dark':
                  ($x['status']==='em_andamento'?'info text-dark':
                  ($x['status']==='finalizado'?'success':'secondary'))
                ?>"><?= e($x['status'] ?: '—') ?></span>
              </td>
              <td><?= $x['executado_por'] ? (int)$x['executado_por'] : '—' ?></td>
              <td><?= e($x['iniciado_em'] ?: '—') ?></td>
              <td><?= e($x['finalizado_em'] ?: '—') ?></td>
              <td class="text-end">
                <?php if ($x['status']==='liberada' && !$x['iniciado_em'] && !$x['finalizado_em']): ?>
                  <form method="post" class="d-inline" onsubmit="return confirm('Cancelar esta liberação? Todas as dependências (respostas/fotos) serão removidas.');">
                    <input type="hidden" name="cancelar_execucao" value="<?= (int)$x['id'] ?>">
                    <button class="btn btn-sm btn-outline-danger">Cancelar</button>
                  </form>
                <?php else: ?>
                  <button class="btn btn-sm btn-outline-secondary" disabled>Cancelar</button>
                <?php endif; ?>
              </td>
            </tr>
          <?php endforeach; ?>
          </tbody>
        </table>
      </div>
    <?php endif; ?>
  </div>
</div>

<!-- MODAL EDITAR (com Gerência e Setor) -->
<div class="modal fade" id="modalEdit" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-lg modal-dialog-scrollable">
    <div class="modal-content">
      <form method="post">
        <input type="hidden" name="edit_pergunta" value="1">
        <input type="hidden" name="pergunta_id" id="edit_pergunta_id" value="">
        <div class="modal-header">
          <h5 class="modal-title">Editar pergunta</h5>
          <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fechar"></button>
        </div>
        <div class="modal-body">
          <div class="row g-3">
            <div class="col-12 col-md-6">
              <label class="form-label">Gerência</label>
              <select name="gerencia_id_edit" id="edit_gerencia" class="form-select" required>
                <option value="">Selecione</option>
                <?php foreach($gerencias as $g): ?>
                  <option value="<?= (int)$g['id'] ?>"><?= e($g['nome']) ?></option>
                <?php endforeach; ?>
              </select>
            </div>
            <div class="col-12 col-md-6">
              <label class="form-label">Setor</label>
              <select name="setor_id_edit" id="edit_setor" class="form-select" required>
                <option value="">Selecione a gerência</option>
              </select>
            </div>
            <div class="col-12">
              <label class="form-label">Texto da pergunta</label>
              <textarea name="texto_edit" id="edit_texto" class="form-control" rows="5" required></textarea>
            </div>
            <div class="col-12 col-md-4">
              <label class="form-label">Peso</label>
              <input type="number" step="1" min="1" name="peso_edit" id="edit_peso" class="form-control" required>
            </div>
            <div class="col-12 col-md-8">
              <label class="form-label">Responsável no setor <span class="text-danger">*</span></label>
              <select name="responsavel_id_edit" id="edit_resp" class="form-select" required>
                <option value="">Selecione</option>
                <?php foreach($colabs as $c): ?>
                  <option value="<?= (int)$c['id'] ?>"><?= e(($c['cargo'] ? $c['cargo'].' - ' : '') . $c['nome']) ?></option>
                <?php endforeach; ?>
              </select>
            </div>
          </div>
        </div>
        <div class="modal-footer">
          <button class="btn btn-primary">Salvar alterações</button>
          <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancelar</button>
        </div>
      </form>
    </div>
  </div>
</div>

<script>
// manter na aba correta via ?tab=
<?php if ($activeTab==='perguntas'): ?>
  if (window.bootstrap && bootstrap.Tab) new bootstrap.Tab(document.getElementById('tab-perguntas')).show();
<?php elseif ($activeTab==='execucoes'): ?>
  if (window.bootstrap && bootstrap.Tab) new bootstrap.Tab(document.getElementById('tab-execs')).show();
<?php endif; ?>

// carregar setores por gerência (form de adicionar)
async function carregarSetores(gerenciaId){
  const sSel = document.getElementById('setor');
  sSel.innerHTML = '<option value="">Carregando...</option>';
  if(!gerenciaId){
    sSel.innerHTML = '<option value="">Selecione a gerência</option>';
    return;
  }
  const res = await fetch('<?= e(url('api/setores_by_gerencia.php')) ?>?gerencia_id='+gerenciaId);
  const js  = await res.json();
  sSel.innerHTML = '<option value="">Selecione</option>';
  js.forEach(s=>{
    const opt=document.createElement('option');
    opt.value=s.id; opt.textContent=s.nome;
    sSel.appendChild(opt);
  });
}
document.getElementById('gerencia')?.addEventListener('change', e=>carregarSetores(e.target.value));

// carregar setores no modal, conforme gerência
async function carregarSetoresEdit(gerenciaId, setorSelecionado){
  const sSel = document.getElementById('edit_setor');
  sSel.innerHTML = '<option value="">Carregando...</option>';
  if(!gerenciaId){
    sSel.innerHTML = '<option value="">Selecione a gerência</option>';
    return;
  }
  const res = await fetch('<?= e(url('api/setores_by_gerencia.php')) ?>?gerencia_id='+gerenciaId);
  const js  = await res.json();
  sSel.innerHTML = '<option value="">Selecione</option>';
  js.forEach(s=>{
    const opt=document.createElement('option');
    opt.value=s.id; opt.textContent=s.nome;
    if (String(s.id) === String(setorSelecionado)) opt.selected = true;
    sSel.appendChild(opt);
  });
}

// popular modal
function fillEditModal(btn){
  const pid  = btn.getAttribute('data-pid');
  const texto= btn.getAttribute('data-texto');
  const peso = btn.getAttribute('data-peso');
  const resp = btn.getAttribute('data-resp');
  const ger  = btn.getAttribute('data-ger');
  const set  = btn.getAttribute('data-set');

  document.getElementById('edit_pergunta_id').value = pid || '';
  document.getElementById('edit_texto').value       = texto || '';
  document.getElementById('edit_peso').value        = peso || '';

  const selResp = document.getElementById('edit_resp');
  if (selResp) selResp.value = resp && resp!=='0' ? resp : '';

  const selGer = document.getElementById('edit_gerencia');
  selGer.value = ger || '';
  carregarSetoresEdit(ger, set);
}

// clique no botão Editar
document.querySelectorAll('.edit-btn').forEach(btn=>{
  btn.addEventListener('click', ()=> fillEditModal(btn));
});

// evento Bootstrap modal
const modal = document.getElementById('modalEdit');
modal.addEventListener('show.bs.modal', function (ev) {
  const btn = ev.relatedTarget;
  if (btn) fillEditModal(btn);
});

// trocar gerência dentro do modal
document.getElementById('edit_gerencia')?.addEventListener('change', function(){
  const newGer = this.value || '';
  carregarSetoresEdit(newGer, null);
});
</script>

<?php include __DIR__ . '/../../app/partials/foot.php'; ?>
