<?php
// public/empresas/novo.php — agora com abas: Info. empresa, Acessos, Extra
require_once __DIR__ . '/../../app/core/auth.php';
require_login();

require_once __DIR__ . '/../../app/config/db.php';
require_once __DIR__ . '/../../app/core/helpers.php';
require_once __DIR__ . '/../../app/core/rbac.php';

require_role('admin'); // somente admin

$pageTitle = 'Nova Empresa / Gerenciar';

function fetch_role_id(PDO $pdo, string $roleName): ?int {
  $st = $pdo->prepare("SELECT id FROM roles WHERE nome=? LIMIT 1");
  $st->execute([$roleName]);
  $id = $st->fetchColumn();
  return $id ? (int)$id : null;
}
function email_exists(PDO $pdo, string $email, int $ignoreId = 0): bool {
  $st = $pdo->prepare("SELECT id FROM usuarios WHERE email=? AND id<>? LIMIT 1");
  $st->execute([$email, $ignoreId]);
  return (bool)$st->fetch();
}
function only_digits($s){ return preg_replace('/\D+/', '', (string)$s); }

/* ===== Utils de paths para arquivos (logo/) ===== */
function dir_public_base(): string {
  return realpath(__DIR__ . '/../../'); // .../public
}
function dir_empresa(int $id): string {
  $d = dir_public_base() . '/uploads/empresas/' . $id;
  if (!is_dir($d)) @mkdir($d, 0775, true);
  return $d;
}
function dir_logo(int $id): string {
  $d = dir_empresa($id) . '/logo';
  if (!is_dir($d)) @mkdir($d, 0775, true);
  return $d;
}
/**
 * Monta a URL pública da logo.
 * - Se $logoName (ou $empresa['logo']) existir, usa ele.
 * - Senão, cai para o primeiro arquivo encontrado na pasta.
 */
function logo_public_url(int $id, ?string $logoName = null): ?string {
  if ($logoName) {
    return url('../uploads/empresas/'.$id.'/logo/'.basename($logoName));
  }
  $dir = dir_logo($id);
  $files = glob($dir.'/*.{png,jpg,jpeg}', GLOB_BRACE);
  if ($files && isset($files[0])) {
    $fname = basename($files[0]);
    return url('uploads/empresas/'.$id.'/logo/'.$fname);
  }
  return null;
}

/* === Carregar empresa (se houver id) === */
$empresaId = isset($_GET['id']) ? (int)$_GET['id'] : null;
$empresa   = null;
if ($empresaId) {
  $st = $pdo->prepare("SELECT * FROM empresas WHERE id=?");
  $st->execute([$empresaId]);
  $empresa = $st->fetch(PDO::FETCH_ASSOC);
  if (!$empresa) $empresaId = null;
}

$activeTab = $_GET['tab'] ?? 'info'; // info | acessos | extra
$errors = [];
$okMsg  = null;

/* ===== POST handlers ===== */
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
  $acao = $_POST['acao'] ?? '';

  // 1) Criar Empresa (tab Info)
  if ($acao === 'criar_empresa') {
    $tipo_pessoa = $_POST['tipo_pessoa'] ?? 'juridica';

    // comuns
    $cep        = trim($_POST['cep'] ?? '');
    $estado     = strtoupper(trim($_POST['estado'] ?? ''));
    $cidade     = trim($_POST['cidade'] ?? '');
    $endereco   = trim($_POST['endereco'] ?? '');
    $numero     = trim($_POST['numero'] ?? '');
    $complemento= trim($_POST['complemento'] ?? '');
    $bairro     = trim($_POST['bairro'] ?? '');
    $contato_nome     = trim($_POST['contato_nome'] ?? '');
    $contato_telefone = trim($_POST['contato_telefone'] ?? '');

    // PJ
    $cnpj           = trim($_POST['cnpj'] ?? '');
    $razao_social   = trim($_POST['razao_social'] ?? '');
    $nome_fantasia  = trim($_POST['nome_fantasia'] ?? '');
    $cnae_codigo    = trim($_POST['cnae_codigo'] ?? '');
    $cnae_descricao = trim($_POST['cnae_descricao'] ?? '');
    $cnpj_status    = trim($_POST['cnpj_status'] ?? '');
    $data_abertura  = trim($_POST['data_abertura'] ?? '');
    // PF
    $cpf            = trim($_POST['cpf'] ?? '');
    $nome_pf        = trim($_POST['nome_pf'] ?? '');

    // valida
    if (!in_array($tipo_pessoa, ['juridica','fisica'], true)) $errors[]='Tipo de pessoa inválido.';
    if ($tipo_pessoa==='juridica') {
      if ($razao_social==='') $errors[]='Razão Social é obrigatória.';
      $cnpj_d = only_digits($cnpj);
      if ($cnpj_d==='' || strlen($cnpj_d)<14) $errors[]='CNPJ inválido/incompleto.';
    } else {
      if ($nome_pf==='') $errors[]='Nome é obrigatório.';
      $cpf_d = only_digits($cpf);
      if ($cpf_d==='' || strlen($cpf_d)<11) $errors[]='CPF inválido/incompleto.';
    }
    if ($cep==='') $errors[]='CEP é obrigatório.';
    if ($estado==='' || strlen($estado)!==2) $errors[]='Estado (UF) é obrigatório.';
    if ($cidade==='') $errors[]='Cidade é obrigatória.';

    if (!$errors) {
      $ins = $pdo->prepare("
        INSERT INTO empresas
          (tipo_pessoa, cnpj, cpf, razao_social, nome_fantasia, nome,
           cnae_codigo, cnae_descricao, cnpj_status, data_abertura,
           cep, estado, cidade, endereco, numero, complemento, bairro,
           contato_nome, contato_telefone, status, criado_em)
        VALUES
          (?, ?, ?, ?, ?, ?,
           ?, ?, ?, ?,
           ?, ?, ?, ?, ?, ?, ?,
           ?, ?, 'ativo', NOW())
      ");
      $ok = $ins->execute([
        $tipo_pessoa,
        $tipo_pessoa==='juridica'? only_digits($cnpj) : null,
        $tipo_pessoa==='fisica'  ? only_digits($cpf)  : null,
        $tipo_pessoa==='juridica'? $razao_social : null,
        $tipo_pessoa==='juridica'? ($nome_fantasia ?: null) : null,
        $tipo_pessoa==='fisica'  ? $nome_pf : null,

        $tipo_pessoa==='juridica'? ($cnae_codigo ?: null) : null,
        $tipo_pessoa==='juridica'? ($cnae_descricao ?: null) : null,
        $tipo_pessoa==='juridica'? ($cnpj_status ?: null) : null,
        ($tipo_pessoa==='juridica' && $data_abertura) ? $data_abertura : null,

        $cep ?: null, $estado ?: null, $cidade ?: null, $endereco ?: null, $numero ?: null, $complemento ?: null, $bairro ?: null,
        $contato_nome ?: null, $contato_telefone ?: null
      ]);
      if ($ok) {
        $newId = (int)$pdo->lastInsertId();
        header('Location: '.url('empresas/novo.php?id='.$newId.'&ok_empresa=1&tab=acessos'));
        exit;
      } else {
        $errors[]='Falha ao salvar a empresa.';
      }
    }
    // rehidrata em caso de erro
    $empresa = [
      'tipo_pessoa'=>$tipo_pessoa,'cnpj'=>$cnpj,'cpf'=>$cpf,
      'razao_social'=>$razao_social,'nome_fantasia'=>$nome_fantasia,'nome'=>$nome_pf,
      'cnae_codigo'=>$cnae_codigo,'cnae_descricao'=>$cnae_descricao,'cnpj_status'=>$cnpj_status,'data_abertura'=>$data_abertura,
      'cep'=>$cep,'estado'=>$estado,'cidade'=>$cidade,'endereco'=>$endereco,'numero'=>$numero,'complemento'=>$complemento,'bairro'=>$bairro,
      'contato_nome'=>$contato_nome,'contato_telefone'=>$contato_telefone,'status'=>'ativo'
    ];
    $activeTab='info';
  }

  // 2) Acessos: criar/editar usuários
  if ($acao === 'salvar_usuario') {
    $activeTab='acessos';
    if (!$empresaId) $errors[]='Crie a empresa primeiro.';
    $userId   = (int)($_POST['user_id'] ?? 0);
    $nome     = trim($_POST['nome'] ?? '');
    $email    = trim($_POST['email'] ?? '');
    $senha    = $_POST['senha'] ?? '';
    $status   = $_POST['status'] ?? 'ativo';
    $roleName = $_POST['role'] ?? '';
    $roleId   = $roleName ? fetch_role_id($pdo, $roleName) : null;

    if ($nome==='') $errors[]='Nome é obrigatório.';
    if ($email==='') $errors[]='Email é obrigatório.';
    if (!$roleId) $errors[]='Perfil inválido.';
    if ($userId<=0 && strlen($senha)<6) $errors[]='Senha deve ter pelo menos 6 caracteres.';
    if (!$errors && email_exists($pdo,$email,$userId)) $errors[]='Este e-mail já está em uso.';

    if (!$errors) {
      if ($userId>0) {
        if ($senha!=='') {
          if (strlen($senha)<6) $errors[]='Senha (nova) deve ter pelo menos 6 caracteres.';
          else {
            $hash=password_hash($senha,PASSWORD_BCRYPT);
            $up=$pdo->prepare("UPDATE usuarios SET nome=?,email=?,senha_hash=?,role_id=?,status=? WHERE id=? AND empresa_id=?");
            $ok=$up->execute([$nome,$email,$hash,$roleId,$status,$userId,$empresaId]);
          }
        } else {
          $up=$pdo->prepare("UPDATE usuarios SET nome=?,email=?,role_id=?,status=? WHERE id=? AND empresa_id=?");
          $ok=$up->execute([$nome,$email,$roleId,$status,$userId,$empresaId]);
        }
        if (empty($errors) && (empty($ok) || !$ok)) $errors[]='Falha ao atualizar o usuário.';
        if (!$errors) { $okMsg='Usuário atualizado com sucesso.'; }
      } else {
        $hash=password_hash($senha,PASSWORD_BCRYPT);
        $ins=$pdo->prepare("INSERT INTO usuarios (nome,email,senha_hash,role_id,empresa_id,status,criado_em) VALUES (?,?,?,?,?, ?,NOW())");
        $ok=$ins->execute([$nome,$email,$hash,$roleId,$empresaId,$status]);
        if ($ok) { $okMsg='Usuário criado com sucesso.'; }
        else { $errors[]='Falha ao criar o usuário.'; }
      }
    }
  }

  // 3) Extra: upload de logo (salva arquivo e grava nome no banco)
  if ($acao === 'upload_logo') {
    $activeTab='extra';
    if (!$empresaId) {
      $errors[]='Crie a empresa primeiro.';
    } else {
      if (!isset($_FILES['logo']) || ($_FILES['logo']['error']??UPLOAD_ERR_NO_FILE)!==UPLOAD_ERR_OK) {
        $errors[]='Selecione uma imagem.';
      } else {
        $f   = $_FILES['logo'];
        $ext = strtolower(pathinfo($f['name'], PATHINFO_EXTENSION));
        if (!in_array($ext, ['png','jpg','jpeg'])) $ext='png';

        $destDir = dir_logo($empresaId);
        // limpa anteriores (mantém 1 só)
        foreach (glob($destDir.'/*') as $old) @unlink($old);

        $fname = 'logo_' . uniqid() . '.' . $ext;
        $dest  = $destDir . '/' . $fname;

        if (!move_uploaded_file($f['tmp_name'], $dest)) {
          $errors[]='Falha ao salvar a imagem.';
        } else {
          // grava o nome no banco
          $upd = $pdo->prepare("UPDATE empresas SET logo=? WHERE id=?");
          if (!$upd->execute([$fname, $empresaId])) {
            $errors[] = 'Logo enviada, mas não foi possível atualizar a empresa no banco.';
          } else {
            $okMsg = 'Logo enviada e gravada com sucesso.';
            // recarrega empresa para refletir a logo atual
            $st = $pdo->prepare("SELECT * FROM empresas WHERE id=?");
            $st->execute([$empresaId]);
            $empresa = $st->fetch(PDO::FETCH_ASSOC) ?: $empresa;
          }
        }
      }
    }
  }
}

/* === Carregar listas de usuários por perfil (Acessos) === */
$byRole = ['gerente'=>[], 'empresa_user'=>[], 'diretor'=>[]];
if ($empresaId) {
  $st = $pdo->prepare("
    SELECT u.id, u.nome, u.email, u.status, r.nome AS role
      FROM usuarios u
      LEFT JOIN roles r ON r.id = u.role_id
     WHERE u.empresa_id = ?
  ORDER BY FIELD(r.nome,'gerente','empresa_user','diretor'), u.nome
  ");
  $st->execute([$empresaId]);
  $rows = $st->fetchAll(PDO::FETCH_ASSOC);
  foreach ($rows as $u) {
    $key = $u['role'] ?: 'outros';
    if (!isset($byRole[$key])) $byRole[$key] = [];
    $byRole[$key][] = $u;
  }
}

include __DIR__ . '/../../app/partials/head.php';
include __DIR__ . '/../../app/partials/topbar.php';

$e = fn($v)=>e($v ?? '');
$tp = $empresa['tipo_pessoa'] ?? 'juridica';
$ufs = ['AC','AL','AM','AP','BA','CE','DF','ES','GO','MA','MG','MS','MT','PA','PB','PE','PI','PR','RJ','RN','RO','RR','RS','SC','SE','SP','TO'];
?>
<div class="container mt-4">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <h1 class="h4 m-0">Empresa & Gestão</h1>
    <div class="d-flex gap-2">
      <a class="btn btn-outline-secondary btn-sm" href="<?= e(url('empresas/index.php')) ?>">Voltar</a>
    </div>
  </div>

  <?php if (isset($_GET['ok_empresa'])): ?>
    <div class="alert alert-success">Empresa criada com sucesso.</div>
  <?php endif; ?>
  <?php if ($okMsg): ?>
    <div class="alert alert-success"><?= e($okMsg) ?></div>
  <?php endif; ?>
  <?php if ($errors): ?>
    <div class="alert alert-danger"><strong>Não foi possível concluir:</strong><br><?= implode('<br>', array_map('e',$errors)) ?></div>
  <?php endif; ?>

  <!-- NAV TABS -->
  <?php
    $disabledTabs = !$empresaId ? 'disabled' : '';
    function tab_url($tab, $empresaId) {
      $base = 'empresas/novo.php';
      $qs = $empresaId ? ('?id='.$empresaId.'&tab='.$tab) : ('?tab='.$tab);
      return url($base.$qs);
    }
  ?>
  <ul class="nav nav-tabs mb-3">
    <li class="nav-item">
      <a class="nav-link <?= $activeTab==='info'?'active':'' ?>" href="<?= e(tab_url('info',$empresaId)) ?>">Info. empresa</a>
    </li>
    <li class="nav-item">
      <a class="nav-link <?= $activeTab==='acessos'?'active':'' ?> <?= $disabledTabs ?>" href="<?= $empresaId? e(tab_url('acessos',$empresaId)) : '#' ?>">Acessos</a>
    </li>
    <li class="nav-item">
      <a class="nav-link <?= $activeTab==='extra'?'active':'' ?> <?= $disabledTabs ?>" href="<?= $empresaId? e(tab_url('extra',$empresaId)) : '#' ?>">Extra</a>
    </li>
  </ul>

  <div class="tab-content">

    <!-- TAB: Info. empresa -->
    <div class="tab-pane fade <?= $activeTab==='info'?'show active':'' ?>">
      <div class="card">
        <div class="card-header fw-bold">Cadastro da Empresa</div>
        <div class="card-body">
          <form method="post" action="<?= e(url('empresas/novo.php' . ($empresaId ? ('?id='.(int)$empresaId.'&tab=info') : '?tab=info'))) ?>" id="formEmpresa" novalidate>
            <input type="hidden" name="acao" value="criar_empresa">

            <!-- Tipo de Pessoa -->
            <div class="mb-2">
              <label class="form-label d-block">Tipo de Pessoa *</label>
              <?php $disabled = $empresaId ? 'disabled' : ''; ?>
              <div class="form-check form-check-inline">
                <input class="form-check-input" type="radio" name="tipo_pessoa" id="tp_jur" value="juridica" <?= $tp==='juridica'?'checked':'' ?> <?= $disabled ?>>
                <label class="form-check-label" for="tp_jur">Pessoa Jurídica</label>
              </div>
              <div class="form-check form-check-inline">
                <input class="form-check-input" type="radio" name="tipo_pessoa" id="tp_fis" value="fisica" <?= $tp==='fisica'?'checked':'' ?> <?= $disabled ?>>
                <label class="form-check-label" for="tp_fis">Pessoa Física</label>
              </div>
              <?php if ($empresaId): ?><div class="form-text">Tipo de pessoa não pode ser alterado aqui.</div><?php endif; ?>
            </div>

            <!-- PJ -->
            <div id="boxPJ" style="<?= $tp==='fisica'?'display:none':'' ?>">
              <div class="row g-3">
                <div class="col-12 col-md-4">
                  <label class="form-label">CNPJ *</label>
                  <div class="input-group">
                    <input name="cnpj" id="cnpj" class="form-control" inputmode="numeric" maxlength="18"
       value="<?= $e($empresa['cnpj'] ?? '') ?>" <?= $disabled ?>>
                    <button type="button" class="btn btn-outline-secondary" id="btnBuscaCNPJ" <?= $disabled ?>>Buscar CNPJ</button>
                  </div>
                  <div class="form-text">Tentaremos preencher via BrasilAPI.</div>
                </div>
                <div class="col-12 col-md-4">
                  <label class="form-label">Razão Social *</label>
                  <input name="razao_social" id="razao_social" class="form-control" value="<?= $e($empresa['razao_social'] ?? '') ?>" <?= $disabled ? 'readonly' : '' ?>>
                </div>
                <div class="col-12 col-md-4">
                  <label class="form-label">Nome Fantasia</label>
                  <input name="nome_fantasia" id="nome_fantasia" class="form-control" value="<?= $e($empresa['nome_fantasia'] ?? '') ?>" <?= $disabled ? 'readonly' : '' ?>>
                </div>
              </div>

              <div class="row g-3 mt-1">
                <div class="col-12 col-md-3">
                  <label class="form-label">Código do CNAE</label>
                  <input name="cnae_codigo" id="cnae_codigo" class="form-control" value="<?= $e($empresa['cnae_codigo'] ?? '') ?>" <?= $disabled ? 'readonly' : '' ?>>
                </div>
                <div class="col-12 col-md-3">
                  <label class="form-label">Status da Empresa</label>
                  <input name="cnpj_status" id="cnpj_status" class="form-control" value="<?= $e($empresa['cnpj_status'] ?? '') ?>" <?= $disabled ? 'readonly' : '' ?>>
                </div>
                <div class="col-12 col-md-3">
                  <label class="form-label">Data de Abertura</label>
                  <input type="date" name="data_abertura" id="data_abertura" class="form-control" value="<?= $e($empresa['data_abertura'] ?? '') ?>" <?= $disabled ? 'readonly' : '' ?>>
                </div>
                <div class="col-12 col-md-3">
                  <label class="form-label">Descrição do CNAE</label>
                  <input name="cnae_descricao" id="cnae_descricao" class="form-control" value="<?= $e($empresa['cnae_descricao'] ?? '') ?>" <?= $disabled ? 'readonly' : '' ?>>
                </div>
              </div>
            </div>

            <!-- PF -->
            <div id="boxPF" style="<?= $tp==='juridica'?'display:none':'' ?>">
              <div class="row g-3">
                <div class="col-12 col-md-4">
                  <label class="form-label">CPF *</label>
                  <input name="cpf" id="cpf" class="form-control" inputmode="numeric" maxlength="14"
       value="<?= $e($empresa['cpf'] ?? '') ?>" <?= $disabled ?>>
                </div>
                <div class="col-12 col-md-8">
                  <label class="form-label">Nome *</label>
                  <input name="nome_pf" id="nome_pf" class="form-control" value="<?= $e($empresa['nome'] ?? '') ?>" <?= $disabled ? 'readonly' : '' ?>>
                </div>
              </div>
            </div>

            <hr class="my-3">
            <!-- Endereço -->
            <div class="row g-3">
              <div class="col-6 col-md-3">
                <label class="form-label">CEP *</label>
                <input name="cep" id="cep" class="form-control" value="<?= $e($empresa['cep'] ?? '') ?>" <?= $empresaId ? 'readonly' : '' ?>>
              </div>
              <div class="col-6 col-md-3">
                <label class="form-label">Estado (UF) *</label>
                <?php $ufAtual = strtoupper($empresa['estado'] ?? ''); ?>
                <select name="estado" id="estado" class="form-select" <?= $empresaId ? 'disabled' : '' ?>>
                  <option value="">Selecione…</option>
                  <?php foreach ($ufs as $uf): ?>
                    <option value="<?= $uf ?>" <?= $ufAtual===$uf?'selected':'' ?>><?= $uf ?></option>
                  <?php endforeach; ?>
                </select>
              </div>
              <div class="col-12 col-md-6">
                <label class="form-label">Cidade *</label>
                <select name="cidade" id="cidade" class="form-select" <?= $empresaId ? 'disabled' : '' ?>>
                  <?php if (!empty($empresa['cidade'])): ?>
                    <option value="<?= $e($empresa['cidade']) ?>" selected><?= $e($empresa['cidade']) ?></option>
                  <?php else: ?>
                    <option value="">Selecione a UF primeiro…</option>
                  <?php endif; ?>
                </select>
              </div>
              <div class="col-12 col-md-6">
                <label class="form-label">Endereço</label>
                <input name="endereco" id="endereco" class="form-control" value="<?= $e($empresa['endereco'] ?? '') ?>" <?= $empresaId ? 'readonly' : '' ?>>
              </div>
              <div class="col-6 col-md-2">
                <label class="form-label">Número</label>
                <input name="numero" id="numero" class="form-control" value="<?= $e($empresa['numero'] ?? '') ?>" <?= $empresaId ? 'readonly' : '' ?>>
              </div>
              <div class="col-6 col-md-4">
                <label class="form-label">Complemento</label>
                <input name="complemento" id="complemento" class="form-control" value="<?= $e($empresa['complemento'] ?? '') ?>" <?= $empresaId ? 'readonly' : '' ?>>
              </div>
              <div class="col-12 col-md-4">
                <label class="form-label">Bairro</label>
                <input name="bairro" id="bairro" class="form-control" value="<?= $e($empresa['bairro'] ?? '') ?>" <?= $empresaId ? 'readonly' : '' ?>>
              </div>
            </div>

            <hr class="my-3">
            <!-- Contato -->
            <div class="row g-3">
              <div class="col-12 col-md-6">
                <label class="form-label">Contato - Nome</label>
                <input name="contato_nome" id="contato_nome" class="form-control" value="<?= $e($empresa['contato_nome'] ?? '') ?>" <?= $empresaId ? 'readonly' : '' ?>>
              </div>
              <div class="col-12 col-md-4">
                <label class="form-label">Contato - Telefone</label>
                <input name="contato_telefone" id="contato_telefone" class="form-control" value="<?= $e($empresa['contato_telefone'] ?? '') ?>" <?= $empresaId ? 'readonly' : '' ?>>
              </div>
            </div>

            <?php if (!$empresaId): ?>
              <div class="mt-3">
                <button class="btn btn-primary">Salvar empresa</button>
              </div>
            <?php else: ?>
              <div class="mt-2 text-muted">Empresa já criada. Use outras abas para continuar.</div>
            <?php endif; ?>
          </form>
        </div>
      </div>
    </div>

    <!-- TAB: Acessos -->
    <div class="tab-pane fade <?= $activeTab==='acessos'?'show active':'' ?>">
      <?php if (!$empresaId): ?>
        <div class="alert alert-info">Crie a empresa na aba "Info. empresa" para liberar os acessos.</div>
      <?php else: ?>
        <?php
          function user_list_block($titulo, $roleKey, $byRole, $empresaId){
            ?>
            <div class="card mb-4">
              <div class="card-header d-flex justify-content-between align-items-center">
                <span class="fw-bold"><?= e($titulo) ?></span>
                <button class="btn btn-sm btn-success"
                        data-bs-toggle="modal"
                        data-bs-target="#modalUsuario"
                        data-mode="create"
                        data-role="<?= e($roleKey) ?>"
                        data-empresa="<?= (int)$empresaId ?>">
                  Novo
                </button>
              </div>
              <div class="card-body">
                <?php if (empty($byRole[$roleKey])): ?>
                  <div class="alert alert-info m-0">Nenhum cadastro.</div>
                <?php else: ?>
                  <div class="list-group">
                    <?php foreach ($byRole[$roleKey] as $u): ?>
                      <div class="list-group-item d-flex justify-content-between align-items-center">
                        <div>
                          <div class="fw-semibold"><?= e($u['nome']) ?></div>
                          <div class="small text-muted"><?= e($u['email']) ?> · Status: <?= e($u['status']) ?></div>
                        </div>
                        <div>
                          <button class="btn btn-sm btn-outline-primary"
                                  data-bs-toggle="modal"
                                  data-bs-target="#modalUsuario"
                                  data-mode="edit"
                                  data-user-id="<?= (int)$u['id'] ?>"
                                  data-nome="<?= e($u['nome']) ?>"
                                  data-email="<?= e($u['email']) ?>"
                                  data-status="<?= e($u['status']) ?>"
                                  data-role="<?= e($roleKey) ?>"
                                  data-empresa="<?= (int)$empresaId ?>">
                            Editar
                          </button>
                        </div>
                      </div>
                    <?php endforeach; ?>
                  </div>
                <?php endif; ?>
              </div>
            </div>
            <?php
          }
          user_list_block('Diretores', 'diretor', $byRole, $empresaId);
          user_list_block('Gerentes', 'gerente', $byRole, $empresaId);
          user_list_block('Responsável Técnico', 'empresa_user', $byRole, $empresaId);
        ?>
      <?php endif; ?>
    </div>

    <!-- TAB: Extra (Upload de Imagem/Logo) -->
    <div class="tab-pane fade <?= $activeTab==='extra'?'show active':'' ?>">
      <?php if (!$empresaId): ?>
        <div class="alert alert-info">Crie a empresa na aba "Info. empresa" para enviar imagens.</div>
      <?php else: ?>
        <?php
          $logoUrl = logo_public_url((int)$empresaId, $empresa['logo'] ?? null);
        ?>
        <div class="card">
          <div class="card-header fw-bold">Logo da Empresa</div>
          <div class="card-body">
            <?php if ($logoUrl): ?>
              <div class="mb-3">
                <img src="<?= e($logoUrl) ?>" alt="logo" style="max-height:80px;border:1px solid #eee;padding:6px;border-radius:8px;background:#fff">
              </div>
            <?php else: ?>
              <div class="mb-3 text-muted">Nenhuma logo enviada ainda.</div>
            <?php endif; ?>

            <form method="post" enctype="multipart/form-data" action="<?= e(url('empresas/novo.php?id='.(int)$empresaId.'&tab=extra')) ?>">
              <input type="hidden" name="acao" value="upload_logo">
              <div class="mb-2">
                <label class="form-label">Enviar imagem (png, jpg, jpeg)</label>
                <input type="file" class="form-control" name="logo" accept="image/*" required>
              </div>
              <button class="btn btn-primary">Enviar</button>
             
            </form>
          </div>
        </div>
      <?php endif; ?>
    </div>

  </div>
</div>

<?php include __DIR__ . '/../../app/partials/foot.php'; ?>

<!-- MODAL: Criar/Editar Usuário (Acessos) -->
<div class="modal fade" id="modalUsuario" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog modal-dialog-centered">
    <form method="post" class="modal-content" action="<?= e(url('empresas/novo.php' . ($empresaId ? ('?id='.(int)$empresaId.'&tab=acessos') : '?tab=acessos'))) ?>">
      <input type="hidden" name="acao" value="salvar_usuario">
      <input type="hidden" name="user_id" id="user_id" value="0">
      <input type="hidden" name="empresa_id" id="empresa_id" value="<?= (int)($empresaId ?? 0) ?>">

      <div class="modal-header">
        <h5 class="modal-title" id="modalUsuarioTitulo">Novo Usuário</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Fechar"></button>
      </div>

      <div class="modal-body">
        <div class="row g-3">
          <div class="col-12">
            <label class="form-label">Perfil</label>
            <select name="role" id="role" class="form-select" required>
              <option value="diretor">Diretor</option>
              <option value="gerente">Gerente</option>
              <option value="empresa_user">Responsável Técnico</option>
            </select>
          </div>
          <div class="col-12">
            <label class="form-label">Nome</label>
            <input name="nome" id="nome" class="form-control" required>
          </div>
          <div class="col-12">
            <label class="form-label">Email</label>
            <input type="email" name="email" id="email" class="form-control" required>
          </div>
          <div class="col-12">
            <label class="form-label">Senha <span class="text-muted small">(deixe em branco para manter)</span></label>
            <input type="password" name="senha" id="senha" class="form-control" minlength="6" placeholder="••••••">
          </div>
          <div class="col-12 col-md-6">
            <label class="form-label">Status</label>
            <select name="status" id="status" class="form-select">
              <option value="ativo">ativo</option>
              <option value="inativo">inativo</option>
            </select>
          </div>
        </div>
      </div>

      <div class="modal-footer">
        <button type="button" class="btn btn-outline-secondary" data-bs-dismiss="modal">Cancelar</button>
        <button class="btn btn-primary">Salvar</button>
      </div>
    </form>
  </div>
</div>

<script>
// Toggle PF/PJ (aba Info)
function updateTipoPessoa() {
  const tp = document.querySelector('input[name="tipo_pessoa"]:checked')?.value || 'juridica';
  document.getElementById('boxPJ').style.display = (tp === 'juridica') ? '' : 'none';
  document.getElementById('boxPF').style.display = (tp === 'fisica') ? '' : 'none';
}
document.querySelectorAll('input[name="tipo_pessoa"]').forEach(r => r.addEventListener('change', updateTipoPessoa));
updateTipoPessoa();

// Busca CNPJ (BrasilAPI)
async function fetchCNPJ(cnpj) {
  const clean = (cnpj || '').replace(/\D+/g,'');
  if (clean.length < 14) { alert('CNPJ incompleto.'); return; }
  const btn = document.getElementById('btnBuscaCNPJ');
  if (btn) { btn.disabled = true; btn.textContent = 'Buscando...'; }
  try {
    const resp = await fetch('https://brasilapi.com.br/api/cnpj/v1/' + clean);
    if (!resp.ok) throw new Error('Falha na consulta do CNPJ');
    const j = await resp.json();
    document.getElementById('razao_social').value   = j.razao_social || j.nome_fantasia || j.nome || '';
    document.getElementById('nome_fantasia').value  = j.nome_fantasia || '';
    document.getElementById('cnae_codigo').value    = (j.cnae_fiscal || (j.cnae && j.cnae.codigo)) || '';
    document.getElementById('cnae_descricao').value = (j.cnae_fiscal_descricao || (j.cnae && j.cnae.descricao)) || '';
    document.getElementById('cnpj_status').value    = j.descricao_situacao_cadastral || j.descricao_situacao || j.situacao || '';
    if (j.data_inicio_atividade) {
      const d = j.data_inicio_atividade; // yyyy-mm-dd
      document.getElementById('data_abertura').value = d.length >= 10 ? d.substring(0,10) : '';
    } else if (j.abertura) {
      const [dd,mm,yy] = (j.abertura||'').split('/');
      if (dd && mm && yy) document.getElementById('data_abertura').value = `${yy}-${mm}-${dd}`;
    }
    if (j.cep) document.getElementById('cep').value = j.cep;
    if (j.uf)  document.getElementById('estado').value = j.uf;
    if (j.municipio) {
      await loadCidades(j.uf);
      document.getElementById('cidade').value = j.municipio;
    }
    if (j.logradouro) document.getElementById('endereco').value = j.logradouro + (j.numero ? (', ' + j.numero) : '');
    if (j.numero)     document.getElementById('numero').value = j.numero;
    if (j.complemento)document.getElementById('complemento').value = j.complemento;
    if (j.bairro)     document.getElementById('bairro').value = j.bairro;
  } catch (e) {
    alert('Não foi possível obter os dados do CNPJ automaticamente.');
  } finally {
    if (btn) { btn.disabled = false; btn.textContent = 'Buscar CNPJ'; }
  }
}
document.getElementById('btnBuscaCNPJ')?.addEventListener('click', () => {
  const cnpj = document.getElementById('cnpj')?.value || '';
  fetchCNPJ(cnpj);
});

// CEP -> ViaCEP
async function fetchCEP(cep) {
  const clean = (cep || '').replace(/\D+/g,'');
  if (clean.length !== 8) return;
  try {
    const r = await fetch('https://viacep.com.br/ws/' + clean + '/json/');
    const j = await r.json();
    if (j && !j.erro) {
      document.getElementById('endereco').value = j.logradouro || '';
      document.getElementById('bairro').value   = j.bairro || '';
      document.getElementById('estado').value   = (j.uf || '').toUpperCase();
      await loadCidades(j.uf);
      document.getElementById('cidade').value   = j.localidade || '';
    }
  } catch(e){}
}
document.getElementById('cep')?.addEventListener('change', (e)=>fetchCEP(e.target.value));

// UF -> Cidades (IBGE)
async function loadCidades(uf) {
  const sel = document.getElementById('cidade');
  if (!sel) return;
  sel.innerHTML = '<option value="">Carregando cidades…</option>';
  if (!uf) { sel.innerHTML = '<option value="">Selecione a UF primeiro…</option>'; return; }
  try {
    const r = await fetch('https://servicodados.ibge.gov.br/api/v1/localidades/estados/' + uf + '/municipios');
    const j = await r.json();
    sel.innerHTML = '<option value="">Selecione…</option>';
    j.sort((a,b)=>a.nome.localeCompare(b.nome)).forEach(m=>{
      const opt = document.createElement('option');
      opt.value = m.nome; opt.textContent = m.nome;
      sel.appendChild(opt);
    });
  } catch(e){
    sel.innerHTML = '<option value="">Falha ao carregar cidades</option>';
  }
}
document.getElementById('estado')?.addEventListener('change', (e)=>loadCidades(e.target.value));

// Modal de usuário (Acessos)
document.addEventListener('show.bs.modal', function(e) {
  const modal = document.getElementById('modalUsuario');
  if (!modal || e.target !== modal) return;

  const btn = e.relatedTarget;
  const mode     = btn?.getAttribute('data-mode') || 'create';
  const role     = btn?.getAttribute('data-role') || 'gerente';
  const empresa  = btn?.getAttribute('data-empresa') || '0';

  const titleEl  = modal.querySelector('#modalUsuarioTitulo');
  const userIdEl = modal.querySelector('#user_id');
  const nomeEl   = modal.querySelector('#nome');
  const emailEl  = modal.querySelector('#email');
  const senhaEl  = modal.querySelector('#senha');
  const statusEl = modal.querySelector('#status');
  const roleEl   = modal.querySelector('#role');
  const empresaEl= modal.querySelector('#empresa_id');

  if (mode === 'edit') {
    titleEl.textContent = 'Editar Usuário';
    userIdEl.value = btn.getAttribute('data-user-id') || '0';
    nomeEl.value   = btn.getAttribute('data-nome') || '';
    emailEl.value  = btn.getAttribute('data-email') || '';
    statusEl.value = btn.getAttribute('data-status') || 'ativo';
    roleEl.value   = btn.getAttribute('data-role') || 'gerente';
    empresaEl.value= empresa;
    // senha em branco mantém atual
    senhaEl.value  = '';
  } else {
    titleEl.textContent = 'Novo Usuário';
    userIdEl.value = '0';
    nomeEl.value   = '';
    emailEl.value  = '';
    statusEl.value = 'ativo';
    roleEl.value   = role;
    empresaEl.value= empresa;
    senhaEl.value  = '';
  }
});

// === Máscara vanilla (sem libs) para CNPJ/CPF ===
(function () {
  function formatByPattern(value, pattern) {
    const digits = (value || '').replace(/\D+/g, '');
    let i = 0, out = '';
    for (const ch of pattern) {
      if (ch === '9') {
        if (i < digits.length) out += digits[i++];
        else break;
      } else {
        if (i === 0 && out === '') continue; // não começa com separador
        if (i <= digits.length) out += ch;
      }
    }
    return out;
  }

  function attachMask(input, pattern, maxLen) {
    if (!input) return;
    if (maxLen) input.setAttribute('maxlength', maxLen);
    input.setAttribute('inputmode', 'numeric');
    const apply = () => { input.value = formatByPattern(input.value, pattern); };
    input.addEventListener('input', apply);
    input.addEventListener('blur', apply);
    input.addEventListener('paste', () => setTimeout(apply, 0));
    // formata o valor inicial (se vier do servidor)
    apply();
  }

  // Encontra campos (existem em PF/PJ alternando visibilidade)
  attachMask(document.getElementById('cnpj'), '99.999.999/9999-99', 18);
  attachMask(document.getElementById('cpf'),  '999.999.999-99',   14);

  // Se você preencher CNPJ/CPF via JS (ex: BrasilAPI), dispare um input para aplicar a máscara:
  window.applyMaskNow = function(id){
    const el = document.getElementById(id);
    if (el) el.dispatchEvent(new Event('input'));
  };
})();
</script>

