<?php
// public/empresa_user/responder.php
require_once __DIR__ . '/../../app/core/auth.php';
require_login();

$u = current_user();
if (($u['role'] ?? '') !== 'empresa_user') {
  redirect('../login.php');
}

require_once __DIR__ . '/../../app/config/db.php';
require_once __DIR__ . '/../../app/core/helpers.php';

$pageTitle = 'Responder Checklist (Passo a passo)';

$empresaId = (int)($u['empresa_id'] ?? 0);
$usuarioId = (int)($u['id'] ?? 0);

/* ===== Descobrir colaborador_id (se houver vínculo) ===== */
$colaboradorId = 0;
if (!empty($_SESSION['user']['colaborador_id'])) {
  $colaboradorId = (int)$_SESSION['user']['colaborador_id'];
}
if ($colaboradorId <= 0) {
  try {
    $q = $pdo->prepare("
      SELECT id FROM colaboradores
       WHERE empresa_id = ? AND (usuario_id = ? OR user_id = ?)
       ORDER BY id LIMIT 1
    ");
    $q->execute([$empresaId, $usuarioId, $usuarioId]);
    $colaboradorId = (int)($q->fetchColumn() ?: 0);
    if ($colaboradorId) $_SESSION['user']['colaborador_id'] = $colaboradorId;
  } catch (Throwable $e) {}
}

/* ===== Carregar/abrir execução ===== */
$execucaoId = (int)($_GET['execucao_id'] ?? 0);
$checklistIdLegacy = (int)($_GET['id'] ?? 0); // suporte link antigo

$exec = null;
if ($execucaoId > 0) {
  $st = $pdo->prepare("
    SELECT x.*, c.titulo
      FROM execucoes x
      JOIN checklists c ON c.id = x.checklist_id
     WHERE x.id = ? AND x.empresa_id = ?
       AND x.status IN ('liberada','em_andamento','finalizado')
     LIMIT 1
  ");
  $st->execute([$execucaoId, $empresaId]);
  $exec = $st->fetch();
} elseif ($checklistIdLegacy > 0) {
  $st = $pdo->prepare("
    SELECT x.*, c.titulo
      FROM execucoes x
      JOIN checklists c ON c.id = x.checklist_id
     WHERE x.checklist_id = ? AND x.empresa_id = ? AND x.status = 'liberada'
  ORDER BY x.id ASC
     LIMIT 1
  ");
  $st->execute([$checklistIdLegacy, $empresaId]);
  $exec = $st->fetch();
}

include __DIR__ . '/../../app/partials/head.php';
include __DIR__ . '/../../app/partials/topbar.php';

if (!$empresaId || !$exec) {
  echo "<div class='container mt-4'>
          <div class='alert alert-danger'>Checklist não encontrado ou não está liberado para sua empresa.</div>
          <a class='btn btn-outline-secondary' href='".e(url('empresa_user/index.php'))."'>Voltar</a>
        </div>";
  include __DIR__ . '/../../app/partials/foot.php';
  exit;
}

/* Se estava 'liberada', ao abrir vira 'em_andamento' */
if ($exec['status'] === 'liberada') {
  $up = $pdo->prepare("
    UPDATE execucoes
       SET status='em_andamento', executado_por=?, iniciado_em=COALESCE(iniciado_em, NOW())
     WHERE id=? AND status='liberada'
  ");
  $up->execute([$usuarioId, (int)$exec['id']]);
  $st = $pdo->prepare("SELECT * FROM execucoes WHERE id=?");
  $st->execute([(int)$exec['id']]);
  $exec = $st->fetch() + $exec;
}

/* Totais (para finalizar quando acabar) */
$totPerg = (int)$pdo->query("SELECT COUNT(*) FROM checklist_perguntas WHERE checklist_id = ".(int)$exec['checklist_id'])->fetchColumn();
$st = $pdo->prepare("SELECT COUNT(*) FROM respostas WHERE execucao_id = ?");
$st->execute([(int)$exec['id']]);
$totResp = (int)$st->fetchColumn();

/* ===== Buscar próxima pergunta (prioriza atribuída ao colaborador) ===== */
$pergunta = null;
if ($colaboradorId > 0) {
  $st = $pdo->prepare("
    SELECT p.id, p.texto, p.peso, p.ordem, g.nome AS gerencia, s.nome AS setor
      FROM checklist_perguntas p
      JOIN checklist_pergunta_responsaveis r ON r.pergunta_id = p.id
      JOIN gerencias g ON g.id = p.gerencia_id
      JOIN setores   s ON s.id = p.setor_id
     WHERE p.checklist_id = ?
       AND r.colaborador_id = ?
       AND NOT EXISTS (
         SELECT 1 FROM respostas rr WHERE rr.execucao_id = ? AND rr.pergunta_id = p.id
       )
  ORDER BY p.ordem, p.id
     LIMIT 1
  ");
  $st->execute([(int)$exec['checklist_id'], (int)$colaboradorId, (int)$exec['id']]);
  $pergunta = $st->fetch();
}
if (!$pergunta) {
  $st = $pdo->prepare("
    SELECT p.id, p.texto, p.peso, p.ordem, g.nome AS gerencia, s.nome AS setor
      FROM checklist_perguntas p
      JOIN gerencias g ON g.id = p.gerencia_id
      JOIN setores   s ON s.id = p.setor_id
     WHERE p.checklist_id = ?
       AND NOT EXISTS (
         SELECT 1 FROM respostas rr WHERE rr.execucao_id = ? AND rr.pergunta_id = p.id
       )
  ORDER BY p.ordem, p.id
     LIMIT 1
  ");
  $st->execute([(int)$exec['checklist_id'], (int)$exec['id']]);
  $pergunta = $st->fetch();
}

/* ===== POST: salvar resposta ===== */
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['responder']) && $pergunta) {
  $resp = $_POST['resp'] ?? '';
  $obs  = trim($_POST['observacao'] ?? '');

  // Validação: resposta escolhida
  if (!in_array($resp, ['conforme','nao_conforme'], true)) {
    $msg = 'Selecione uma opção (Conforme / Não conforme).';
  }

  // Validação: pelo menos 1 foto enviada (obrigatório)
  if (empty($msg)) {
    $temFotoOk = false;
    if (!empty($_FILES['fotos']) && is_array($_FILES['fotos']['name'])) {
      $errs = $_FILES['fotos']['error'];
      $tmps = $_FILES['fotos']['tmp_name'];
      for ($i=0; $i<count($tmps); $i++) {
        if (($errs[$i] ?? UPLOAD_ERR_NO_FILE) === UPLOAD_ERR_OK && is_uploaded_file($tmps[$i])) {
          $temFotoOk = true; break;
        }
      }
    }
    if (!$temFotoOk) $msg = 'Envie pelo menos 1 foto.';
  }

  if (!empty($msg)) {
    echo "<div class='container mt-3'><div class='alert alert-danger'>".e($msg)."</div></div>";
  } else {
    // grava resposta
    $ins = $pdo->prepare("
      INSERT INTO respostas (execucao_id, pergunta_id, resposta, observacao, criado_em)
      VALUES (?, ?, ?, ?, NOW())
    ");
    $ok  = $ins->execute([(int)$exec['id'], (int)$pergunta['id'], $resp, $obs]);

    if ($ok) {
      $respostaId = (int)$pdo->lastInsertId();

      // upload de fotos (obrigatório já validado)
      if (!empty($_FILES['fotos']) && is_array($_FILES['fotos']['name'])) {
        $basePublic = realpath(__DIR__ . '/../../'); // .../public
        $dir = $basePublic . '/uploads/execucoes/' . (int)$exec['id'] . '/pergunta_' . (int)$pergunta['id'];
        if (!is_dir($dir)) @mkdir($dir, 0775, true);

        $names = $_FILES['fotos']['name'];
        $tmps  = $_FILES['fotos']['tmp_name'];
        $errs  = $_FILES['fotos']['error'];

        for ($i=0; $i<count($names); $i++) {
          if (($errs[$i] ?? UPLOAD_ERR_NO_FILE) !== UPLOAD_ERR_OK) continue;
          if (!is_uploaded_file($tmps[$i])) continue;
          $ext = strtolower(pathinfo($names[$i], PATHINFO_EXTENSION));
          if (!in_array($ext, ['png','jpg','jpeg','webp'])) $ext = 'jpg';
          $dest = $dir . '/foto_' . uniqid('', true) . '.' . $ext;
          if (@move_uploaded_file($tmps[$i], $dest)) {
            $rel = 'uploads/execucoes/' . (int)$exec['id'] . '/pergunta_' . (int)$pergunta['id'] . '/' . basename($dest);
            $pdo->prepare("INSERT INTO resposta_fotos (resposta_id, caminho) VALUES (?, ?)")->execute([$respostaId, $rel]);
          }
        }
      }

      // próxima pergunta
      header("Location: ".url('empresa_user/responder.php?execucao_id='.(int)$exec['id']));
      exit;
    } else {
      echo "<div class='container mt-3'><div class='alert alert-danger'>Falha ao salvar a resposta.</div></div>";
    }
  }
}

/* ===== Finalização automática quando acabar ===== */
if (!$pergunta) {
  $st = $pdo->prepare("SELECT COUNT(*) FROM respostas WHERE execucao_id = ?");
  $st->execute([(int)$exec['id']]);
  $totResp = (int)$st->fetchColumn();

  if ($totPerg > 0 && $totResp >= $totPerg) {
    if ($exec['status'] !== 'finalizado') {
      $pdo->prepare("
        UPDATE execucoes
           SET status='finalizado', finalizado_em=NOW()
         WHERE id=? AND status IN ('em_andamento','liberada')
      ")->execute([(int)$exec['id']]);
      $exec['status'] = 'finalizado';
    }

    echo "<div class='container mt-4'>
            <div class='card'>
              <div class='card-body'>
                <h5 class='card-title'>Execução concluída!</h5>
                <p>Todas as perguntas desta execução foram respondidas.</p>
                <div class='d-flex gap-2'>
                  <a class='btn btn-outline-secondary' href='".e(url('empresa_user/index.php'))."'>Voltar</a>
                </div>
              </div>
            </div>
          </div>";
    include __DIR__ . '/../../app/partials/foot.php';
    exit;
  } else {
    echo "<div class='container mt-4'>
            <div class='alert alert-info'>
              Não há pergunta disponível para você agora.
              <br><small>Respondidas: <strong>".(int)$totResp."</strong> de <strong>".(int)$totPerg."</strong>.</small>
            </div>
            <a class='btn btn-outline-secondary' href='".e(url('empresa_user/index.php'))."'>Voltar</a>
          </div>";
    include __DIR__ . '/../../app/partials/foot.php';
    exit;
  }
}
?>
<style>
/* Layout */
.step-card{max-width:980px;margin:0 auto}
.big-q{font-weight:700;font-size:1.05rem}

/* ===== Segmented (Conforme / Não conforme) ===== */
.segmented{
  display:inline-flex; gap:0; border:1px solid #e5e7eb; border-radius:14px; overflow:hidden;
  background:#fff; box-shadow:0 2px 10px rgba(0,0,0,.04);
}
.segmented-input{ position:absolute; opacity:0; pointer-events:none }
.segmented-btn{
  --pad-x: 1.1rem; --pad-y:.7rem;
  display:flex; align-items:center; gap:.55rem;
  padding:var(--pad-y) var(--pad-x); cursor:pointer; user-select:none;
  font-weight:700; letter-spacing:.2px; transition: all .15s ease;
  background:linear-gradient(#fff,#fafafa);
}
.segmented-btn:hover{ background:linear-gradient(#fafafa,#f5f5f5) }
.segmented-btn + .segmented-btn{ border-left:1px solid #e5e7eb }

.segmented-icon{
  width:28px;height:28px;border-radius:50%;
  display:grid;place-items:center;font-size:16px;
  transition:transform .15s ease, box-shadow .15s ease, background .15s ease, color .15s ease;
}

/* estado selecionado */
#opt-conforme:checked + .segmented-ok,
#opt-nao:checked + .segmented-nok{
  color:#111827; background:#fff; position:relative;
}
#opt-conforme:checked + .segmented-ok::after,
#opt-nao:checked + .segmented-nok::after{
  content:""; position:absolute; inset:0; pointer-events:none;
  box-shadow:inset 0 0 0 2px rgba(13,110,253,.18), 0 6px 20px rgba(0,0,0,.06);
  border-radius:14px;
}

/* OK */
.segmented-ok .segmented-icon{
  background:#d1fae5; color:#0f766e;
}
#opt-conforme:checked + .segmented-ok .segmented-icon{
  background:#10b981; color:#ffffff; transform:scale(1.05);
  box-shadow:0 0 0 6px rgba(16,185,129,.15);
}

/* NOK */
.segmented-nok .segmented-icon{
  background:#fee2e2; color:#b91c1c;
}
#opt-nao:checked + .segmented-nok .segmented-icon{
  background:#ef4444; color:#ffffff; transform:scale(1.05);
  box-shadow:0 0 0 6px rgba(239,68,68,.15);
}

/* ===== Botões Ação ===== */
.btn-ico{ margin-left:.4rem; display:inline-block; transform:translateY(1px) }
.btn-gradient-primary{
  --g1:#2563eb; --g2:#7c3aed;
  border:none; color:#fff; font-weight:700; letter-spacing:.2px;
  padding:.7rem 1.1rem; border-radius:12px;
  background: linear-gradient(135deg, var(--g1), var(--g2));
  box-shadow: 0 8px 24px rgba(37,99,235,.25);
  transition: transform .08s ease, box-shadow .12s ease, filter .12s ease;
}
.btn-gradient-primary:hover{ transform: translateY(-1px); filter: brightness(1.05) }
.btn-gradient-primary:active{ transform: translateY(0); box-shadow: 0 6px 18px rgba(37,99,235,.22) }

.btn-soft-secondary{
  border:1px solid #e5e7eb; color:#374151; background:#fff;
  padding:.7rem 1rem; border-radius:12px; font-weight:600;
  transition: background .12s ease, transform .08s ease, box-shadow .12s ease;
}
.btn-soft-secondary:hover{ background:#f9fafb; transform:translateY(-1px); box-shadow:0 6px 16px rgba(0,0,0,.06) }
.btn-soft-secondary:active{ transform:translateY(0) }

/* ===== Pré-visualização de fotos com remoção ===== */
.preview-grid{
  display:grid; grid-template-columns:repeat(auto-fill,minmax(120px,1fr));
  gap:10px; margin-top:10px;
}
.preview-card{
  position:relative; border:1px solid #e5e7eb; border-radius:10px; overflow:hidden;
  background:#f8fafc; aspect-ratio:1/1; display:flex; align-items:center; justify-content:center;
}
.preview-card img{ width:100%; height:100%; object-fit:cover; display:block; }
.preview-remove{
  position:absolute; top:6px; right:6px; border:none; border-radius:999px;
  width:28px; height:28px; display:grid; place-items:center; cursor:pointer;
  background:#ffffffdd; color:#ef4444; font-weight:700;
  box-shadow:0 2px 8px rgba(0,0,0,.1);
}
.preview-remove:hover{ background:#fff; }
.preview-hint{ font-size:.85rem; color:#6b7280 }
</style>

<div class="container my-4">
  <div class="d-flex justify-content-between align-items-center mb-3">
    <div>
      <h1 class="h5 m-0"><?= e($exec['titulo'] ?? ('Execução #'.(int)$exec['id'])) ?></h1>
      <div class="small text-muted">
        Execução #<?= (int)$exec['id'] ?> · Status:
        <span class="badge <?= $exec['status']==='finalizado'?'bg-success':($exec['status']==='em_andamento'?'bg-warning text-dark':'bg-secondary') ?>">
          <?= e($exec['status']) ?>
        </span>
      </div>
    </div>
    <a class="btn btn-outline-secondary" href="<?= e(url('empresa_user/index.php')) ?>">Sair</a>
  </div>

  <div class="card step-card">
    <div class="card-body">
      <div class="d-flex justify-content-between align-items-start">
        <div>
          <div class="text-muted small mb-1">
            Gerência: <strong><?= e($pergunta['gerencia']) ?></strong> ·
            Setor: <strong><?= e($pergunta['setor']) ?></strong>
          </div>
          <div class="big-q mb-2"><?= nl2br(e($pergunta['texto'])) ?></div>
        </div>
      </div>

      <form method="post" enctype="multipart/form-data" class="mt-3" id="responderForm">
        <input type="hidden" name="responder" value="1">

        <!-- escolhas (OBRIGATÓRIO) — versão moderna -->
        <div class="mb-3">
          <div class="segmented" role="radiogroup" aria-label="Escolha se está conforme">
            <input type="radio" class="segmented-input" name="resp" id="opt-conforme" value="conforme" required>
            <label class="segmented-btn segmented-ok" for="opt-conforme" tabindex="0" aria-label="Conforme">
              <span class="segmented-icon">✓</span>
              <span class="segmented-text">Conforme</span>
            </label>

            <input type="radio" class="segmented-input" name="resp" id="opt-nao" value="nao_conforme" required>
            <label class="segmented-btn segmented-nok" for="opt-nao" tabindex="0" aria-label="Não conforme">
              <span class="segmented-icon">!</span>
              <span class="segmented-text">Não conforme</span>
            </label>
          </div>
        </div>

        <!-- observação + ditado -->
        <div class="mb-3">
          <label class="form-label">Observação</label>
          <div class="input-group">
            <textarea name="observacao" id="obs" class="form-control" rows="3" placeholder="Digite (ou use ditado)…"></textarea>
            <button type="button" class="btn btn-outline-secondary" id="btn-dict">🎙️ Ditado</button>
          </div>
          <div class="form-text">Use o ditado por voz para preencher automaticamente.</div>
        </div>

        <!-- fotos (OBRIGATÓRIO) com preview/remover -->
        <div class="mb-2">
          <label class="form-label">Fotos (obrigatório ao menos 1)</label>
          <input type="file" id="fotos" name="fotos[]" class="form-control" accept="image/*" capture="environment" multiple required>
          <div class="preview-hint mt-1">Você pode tirar fotos na hora (câmera) ou anexar da galeria. Toque no <strong>×</strong> para remover antes de enviar.</div>
          <div class="preview-grid" id="previewGrid"></div>
        </div>

        <div class="d-flex justify-content-between align-items-center gap-2 mt-3">
          <a class="btn btn-soft-secondary" href="<?= e(url('empresa_user/index.php')) ?>">
            <span class="btn-ico">↩</span>
            Cancelar
          </a>
          <button class="btn btn-gradient-primary">
            Salvar e continuar
            <span class="btn-ico">▶</span>
          </button>
        </div>
      </form>
    </div>
  </div>
</div>

<script>
// Ditado por voz (Web Speech API - Chrome)
(function(){
  const btn = document.getElementById('btn-dict');
  const ta  = document.getElementById('obs');
  if (!btn || !ta) return;

  const Recognition = window.SpeechRecognition || window.webkitSpeechRecognition;
  if (!Recognition) {
    btn.addEventListener('click', ()=> alert('Seu navegador não suporta ditado por voz.'));
    return;
  }

  let rec, listening = false;
  function start(){
    if (listening) return;
    rec = new Recognition();
    rec.lang = 'pt-BR';
    rec.continuous = false;
    rec.interimResults = true;
    listening = true;
    btn.disabled = true; btn.textContent = '🎙️ Ouvindo…';
    let finalTxt = ta.value ? (ta.value + ' ') : '';
    rec.onresult = (e)=>{
      let interim = '';
      for (let i=e.resultIndex; i<e.results.length; i++){
        const tr = e.results[i][0].transcript;
        if (e.results[i].isFinal) finalTxt += tr + ' ';
        else interim += tr;
      }
      ta.value = finalTxt + interim;
    };
    rec.onend = ()=>{ listening=false; btn.disabled=false; btn.textContent='🎙️ Ditado'; };
    rec.onerror = ()=>{ listening=false; btn.disabled=false; btn.textContent='🎙️ Ditado'; alert('Falha no reconhecimento de voz.'); };
    rec.start();
  }
  btn.addEventListener('click', start);
})();

// Pré-visualização + remoção individual das fotos selecionadas
(function(){
  const input = document.getElementById('fotos');
  const grid  = document.getElementById('previewGrid');
  if (!input || !grid) return;

  // estado atual dos arquivos selecionados (espelho do FileList)
  let files = [];

  function syncInputFiles(){
    const dt = new DataTransfer();
    files.forEach(f => dt.items.add(f));
    input.files = dt.files;
  }

  function renderPreviews(){
    grid.innerHTML = '';
    files.forEach((file, idx) => {
      const url = URL.createObjectURL(file);
      const card = document.createElement('div');
      card.className = 'preview-card';
      card.innerHTML = `
        <img src="${url}" alt="foto">
        <button type="button" class="preview-remove" aria-label="Remover foto" data-idx="${idx}">×</button>
      `;
      grid.appendChild(card);
      // revoga URL ao descarregar
      card.querySelector('img').addEventListener('load', ()=> URL.revokeObjectURL(url));
    });
  }

  input.addEventListener('change', (e) => {
    const selected = Array.from(e.target.files || []);
    // concatena (permite adicionar mais depois)
    files = files.concat(selected);
    syncInputFiles();
    renderPreviews();
  });

  grid.addEventListener('click', (e) => {
    const btn = e.target.closest('.preview-remove');
    if (!btn) return;
    const idx = parseInt(btn.getAttribute('data-idx'), 10);
    if (!isNaN(idx)) {
      files.splice(idx, 1);
      syncInputFiles();
      renderPreviews();
    }
  });

  // Validação: exigir ao menos 1 foto no submit
  const form = document.getElementById('responderForm');
  form.addEventListener('submit', (ev) => {
    if (files.length === 0) {
      ev.preventDefault();
      alert('Envie pelo menos 1 foto para continuar.');
      input.focus();
    }
  });
})();
</script>

<?php include __DIR__ . '/../../app/partials/foot.php'; ?>
